@extends('components.app-layout')

@section('title', 'Inventory Items')

@php
    $dynamicSubtitle = 'Manage ' . ($organizationSettings['name'] ?? 'Presbyterian Church of Ghana') . ' inventory items';
@endphp

@section('subtitle', $dynamicSubtitle)

@section('content')
<div class="min-h-screen bg-gradient-to-br from-slate-50 via-teal-50 to-cyan-50 py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 space-y-8">
        
        <!-- Header Section -->
        <div class="relative overflow-hidden bg-gradient-to-r from-teal-600 via-cyan-600 to-blue-600 rounded-3xl shadow-2xl">
            <div class="absolute inset-0 bg-black opacity-5"></div>
            <div class="absolute top-0 right-0 w-64 h-64 bg-white opacity-5 rounded-full transform translate-x-32 -translate-y-32"></div>
            <div class="absolute bottom-0 left-0 w-48 h-48 bg-white opacity-5 rounded-full transform -translate-x-24 translate-y-24"></div>
            
            <div class="relative px-8 py-16">
                <div class="max-w-4xl mx-auto text-center">
                    <div class="inline-flex items-center justify-center w-20 h-20 bg-white bg-opacity-20 backdrop-blur-sm rounded-2xl mb-6 shadow-xl">
                        <i class="fas fa-boxes text-3xl text-white"></i>
                    </div>
                    <h1 class="text-5xl font-bold text-white mb-4">Inventory Items</h1>
                    <p class="text-xl text-white opacity-90 mb-8 max-w-2xl mx-auto">
                        Browse, search, and manage all your inventory items in one place
                    </p>
                    <div class="flex items-center justify-center space-x-8 text-white/80">
                        <div class="flex items-center">
                            <i class="fas fa-search mr-2"></i>
                            <span class="text-sm">Advanced Search</span>
                        </div>
                        <div class="flex items-center">
                            <i class="fas fa-filter mr-2"></i>
                            <span class="text-sm">Smart Filtering</span>
                        </div>
                        <div class="flex items-center">
                            <i class="fas fa-chart-line mr-2"></i>
                            <span class="text-sm">Stock Tracking</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Search and Filters -->
        <div class="bg-white/90 backdrop-blur-sm rounded-2xl shadow-xl border border-gray-100/50 p-8">
            <form method="GET" action="{{ route('inventory.items.index') }}" class="space-y-6">
                <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
                    <!-- Search -->
                    <div class="md:col-span-2">
                        <label for="search" class="block text-sm font-medium text-gray-700 mb-2">Search Items</label>
                        <div class="relative">
                            <input type="text" name="search" id="search" value="{{ request('search') }}" 
                                   class="block w-full px-4 py-3 pl-12 bg-white/50 backdrop-blur-sm border-2 border-gray-200/50 rounded-2xl focus:ring-4 focus:ring-teal-500/20 focus:border-teal-500 transition-all duration-300 text-gray-900 font-medium shadow-sm hover:shadow-md placeholder-gray-400" 
                                   placeholder="Search by name, code, or description...">
                            <div class="absolute inset-y-0 left-0 flex items-center pl-4">
                                <i class="fas fa-search text-gray-400"></i>
                            </div>
                        </div>
                    </div>

                    <!-- Category Filter -->
                    <div>
                        <label for="category" class="block text-sm font-medium text-gray-700 mb-2">Category</label>
                        <div class="relative">
                            <select name="category" id="category" class="block w-full px-4 py-3 bg-white/50 backdrop-blur-sm border-2 border-gray-200/50 rounded-2xl focus:ring-4 focus:ring-teal-500/20 focus:border-teal-500 transition-all duration-300 text-gray-900 font-medium shadow-sm hover:shadow-md appearance-none">
                                <option value="">All Categories</option>
                                @foreach($categories as $category)
                                    <option value="{{ $category->id }}" {{ request('category') == $category->id ? 'selected' : '' }}>
                                        {{ $category->name }}
                                    </option>
                                @endforeach
                            </select>
                            <div class="absolute inset-y-0 right-0 flex items-center pr-4 pointer-events-none">
                                <i class="fas fa-chevron-down text-gray-400"></i>
                            </div>
                        </div>
                    </div>

                    <!-- Stock Status Filter -->
                    <div>
                        <label for="stock_status" class="block text-sm font-medium text-gray-700 mb-2">Stock Status</label>
                        <div class="relative">
                            <select name="stock_status" id="stock_status" class="block w-full px-4 py-3 bg-white/50 backdrop-blur-sm border-2 border-gray-200/50 rounded-2xl focus:ring-4 focus:ring-teal-500/20 focus:border-teal-500 transition-all duration-300 text-gray-900 font-medium shadow-sm hover:shadow-md appearance-none">
                                <option value="">All Stock Levels</option>
                                <option value="in_stock" {{ request('stock_status') == 'in_stock' ? 'selected' : '' }}>In Stock</option>
                                <option value="low_stock" {{ request('stock_status') == 'low_stock' ? 'selected' : '' }}>Low Stock</option>
                                <option value="out_of_stock" {{ request('stock_status') == 'out_of_stock' ? 'selected' : '' }}>Out of Stock</option>
                            </select>
                            <div class="absolute inset-y-0 right-0 flex items-center pr-4 pointer-events-none">
                                <i class="fas fa-chevron-down text-gray-400"></i>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="flex items-center justify-between">
                    <a href="{{ route('inventory.items.index') }}" class="text-sm text-gray-600 hover:text-gray-800 font-medium">
                        <i class="fas fa-times mr-2"></i>
                        Clear Filters
                    </a>
                    <button type="submit" class="px-6 py-3 bg-gradient-to-r from-teal-600 via-cyan-600 to-blue-600 text-white font-bold rounded-xl hover:from-teal-700 hover:via-cyan-700 hover:to-blue-700 transition-all duration-300 shadow-lg hover:shadow-xl transform hover:scale-105">
                        <i class="fas fa-search mr-2"></i>
                        Search Items
                    </button>
                </div>
            </form>
        </div>

        <!-- Quick Actions -->
        <div class="flex flex-col lg:flex-row justify-between items-start lg:items-center gap-6">
            <div class="space-y-3">
                <div class="flex items-center space-x-4">
                    <div class="w-12 h-12 bg-gradient-to-r from-teal-600 to-cyan-600 rounded-2xl flex items-center justify-center shadow-xl">
                        <i class="fas fa-boxes text-white text-xl"></i>
                    </div>
                    <div>
                        <h2 class="text-3xl font-bold bg-gradient-to-r from-teal-600 via-cyan-600 to-blue-600 bg-clip-text text-transparent">Inventory Items</h2>
                        <p class="text-gray-600 text-lg mt-1">{{ $items->total() }} items found</p>
                    </div>
                </div>
            </div>
            <div class="flex flex-wrap items-center gap-4">
                <a href="{{ route('inventory.items.create') }}" class="inline-flex items-center px-6 py-3 bg-gradient-to-r from-emerald-600 via-green-600 to-teal-600 text-white font-bold rounded-xl hover:from-emerald-700 hover:via-green-700 hover:to-teal-700 transition-all duration-300 shadow-xl hover:shadow-2xl transform hover:scale-105">
                    <i class="fas fa-plus mr-2"></i>
                    Add Item
                </a>
                <a href="{{ route('inventory.categories.index') }}" class="inline-flex items-center px-4 py-3 bg-white text-gray-700 font-medium rounded-xl border border-gray-200 hover:bg-gray-50 transition-all duration-200 shadow-md hover:shadow-lg transform hover:scale-105">
                    <i class="fas fa-tags mr-2"></i>
                    Categories
                </a>
                <a href="{{ route('inventory.alerts') }}" class="inline-flex items-center px-4 py-3 bg-white text-gray-700 font-medium rounded-xl border border-gray-200 hover:bg-gray-50 transition-all duration-200 shadow-md hover:shadow-lg transform hover:scale-105">
                    <i class="fas fa-exclamation-triangle mr-2"></i>
                    Alerts
                </a>
            </div>
        </div>

        <!-- Items Grid -->
        @if($items->count() > 0)
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                @foreach($items as $item)
                    <div class="group bg-white/90 backdrop-blur-sm rounded-2xl shadow-xl border border-gray-100/50 p-6 hover:shadow-2xl transition-all duration-300 transform hover:scale-105">
                        <!-- Item Header -->
                        <div class="flex items-start justify-between mb-4">
                            <div class="flex items-center space-x-3">
                                <div class="w-12 h-12 rounded-xl flex items-center justify-center shadow-lg" style="background: linear-gradient(135deg, {{ $item->category->color ?? '#6B7280' }}, {{ $item->category->color ?? '#6B7280' }}99)">
                                    <i class="{{ $item->category->icon ?? 'fas fa-box' }} text-white text-lg"></i>
                                </div>
                                <div class="flex-1">
                                    <h3 class="font-bold text-gray-900 group-hover:text-teal-600 transition-colors duration-200 line-clamp-2">{{ $item->name }}</h3>
                                    <p class="text-sm text-gray-500 font-medium">{{ $item->item_code }}</p>
                                </div>
                            </div>
                            
                            <!-- Stock Status Badge -->
                            @php
                                $stockStatus = $item->stock_status;
                                $badgeClass = match($stockStatus) {
                                    'out_of_stock' => 'bg-red-100 text-red-800',
                                    'low_stock' => 'bg-orange-100 text-orange-800',
                                    'overstock' => 'bg-blue-100 text-blue-800',
                                    default => 'bg-green-100 text-green-800'
                                };
                                $badgeIcon = match($stockStatus) {
                                    'out_of_stock' => 'fas fa-times-circle',
                                    'low_stock' => 'fas fa-exclamation-triangle',
                                    'overstock' => 'fas fa-arrow-up',
                                    default => 'fas fa-check-circle'
                                };
                            @endphp
                            <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium {{ $badgeClass }}">
                                <i class="{{ $badgeIcon }} mr-1"></i>
                                {{ ucfirst(str_replace('_', ' ', $stockStatus)) }}
                            </span>
                        </div>

                        <!-- Category and Location -->
                        <div class="mb-4">
                            <div class="flex items-center text-sm text-gray-600 mb-1">
                                <i class="fas fa-tag mr-2"></i>
                                {{ $item->category->name ?? 'No Category' }}
                            </div>
                            @if($item->location)
                                <div class="flex items-center text-sm text-gray-600">
                                    <i class="fas fa-map-marker-alt mr-2"></i>
                                    {{ $item->location }}
                                </div>
                            @endif
                        </div>

                        <!-- Stock Information -->
                        <div class="bg-gray-50 rounded-xl p-4 mb-4">
                            <div class="flex items-center justify-between mb-2">
                                <span class="text-sm font-medium text-gray-600">Current Stock</span>
                                <span class="text-lg font-bold text-gray-900">{{ $item->current_stock }}</span>
                            </div>
                            <div class="flex items-center justify-between text-sm text-gray-500">
                                <span>Min: {{ $item->minimum_stock }}</span>
                                <span>{{ $item->unit_of_measure }}</span>
                            </div>
                            
                            <!-- Stock Progress Bar -->
                            @php
                                $maxForProgress = $item->maximum_stock ?? ($item->minimum_stock * 3);
                                $percentage = $maxForProgress > 0 ? min(($item->current_stock / $maxForProgress) * 100, 100) : 0;
                            @endphp
                            <div class="w-full bg-gray-200 rounded-full h-2 mt-2">
                                <div class="h-2 rounded-full transition-all duration-500 {{ $stockStatus === 'out_of_stock' ? 'bg-red-500' : ($stockStatus === 'low_stock' ? 'bg-orange-500' : 'bg-emerald-500') }}" style="width: {{ $percentage }}%"></div>
                            </div>
                        </div>

                        <!-- Cost Information -->
                        <div class="flex items-center justify-between mb-4 text-sm">
                            <span class="text-gray-600">Unit Cost:</span>
                            <span class="font-medium text-gray-900">₵{{ number_format($item->unit_cost, 2) }}</span>
                        </div>
                        <div class="flex items-center justify-between mb-4 text-sm">
                            <span class="text-gray-600">Stock Value:</span>
                            <span class="font-bold text-teal-600">₵{{ number_format($item->stock_value, 2) }}</span>
                        </div>

                        <!-- Expiry Information -->
                        @if($item->track_expiry && $item->expiry_date)
                            @php
                                $daysToExpiry = $item->days_to_expiry;
                            @endphp
                            @if($daysToExpiry <= 30)
                                <div class="mb-4">
                                    @if($daysToExpiry <= 7)
                                        <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800">
                                            <i class="fas fa-exclamation-triangle mr-1"></i>
                                            Expires in {{ $daysToExpiry }} days
                                        </span>
                                    @elseif($daysToExpiry <= 14)
                                        <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-orange-100 text-orange-800">
                                            <i class="fas fa-clock mr-1"></i>
                                            Expires in {{ $daysToExpiry }} days
                                        </span>
                                    @else
                                        <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
                                            <i class="fas fa-calendar mr-1"></i>
                                            Expires in {{ $daysToExpiry }} days
                                        </span>
                                    @endif
                                </div>
                            @endif
                        @endif

                        <!-- Action Buttons -->
                        <div class="flex space-x-2">
                            <a href="{{ route('inventory.items.show', $item) }}" class="flex-1 bg-teal-600 text-white px-3 py-2 rounded-lg text-sm font-medium hover:bg-teal-700 transition-colors text-center">
                                View Details
                            </a>
                            <button onclick="showStockModal({{ $item->id }}, '{{ $item->name }}', {{ $item->current_stock }})" class="flex-1 bg-emerald-600 text-white px-3 py-2 rounded-lg text-sm font-medium hover:bg-emerald-700 transition-colors">
                                Update Stock
                            </button>
                        </div>
                    </div>
                @endforeach
            </div>

            <!-- Pagination -->
            @if($items->hasPages())
                <div class="flex justify-center">
                    {{ $items->appends(request()->query())->links() }}
                </div>
            @endif
        @else
            <!-- No Items Found -->
            <div class="bg-white/90 backdrop-blur-sm rounded-2xl shadow-xl border border-gray-100/50 p-12 text-center">
                <div class="w-20 h-20 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-6">
                    <i class="fas fa-search text-gray-400 text-3xl"></i>
                </div>
                <h2 class="text-2xl font-bold text-gray-900 mb-4">No Items Found</h2>
                <p class="text-gray-600 mb-6">
                    @if(request()->hasAny(['search', 'category', 'stock_status']))
                        No items match your current filters. Try adjusting your search criteria.
                    @else
                        You haven't added any inventory items yet. Start by adding your first item.
                    @endif
                </p>
                <div class="flex justify-center space-x-4">
                    @if(request()->hasAny(['search', 'category', 'stock_status']))
                        <a href="{{ route('inventory.items.index') }}" class="inline-flex items-center px-6 py-3 bg-gray-600 text-white font-medium rounded-xl hover:bg-gray-700 transition-colors">
                            <i class="fas fa-times mr-2"></i>
                            Clear Filters
                        </a>
                    @endif
                    <a href="{{ route('inventory.items.create') }}" class="inline-flex items-center px-6 py-3 bg-emerald-600 text-white font-medium rounded-xl hover:bg-emerald-700 transition-colors">
                        <i class="fas fa-plus mr-2"></i>
                        Add First Item
                    </a>
                </div>
            </div>
        @endif
    </div>
</div>

<!-- Stock Update Modal -->
<div id="stockModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 flex items-center justify-center">
    <div class="bg-white rounded-2xl shadow-2xl max-w-md w-full mx-4">
        <div class="p-6">
            <div class="flex items-center justify-between mb-6">
                <h3 class="text-xl font-bold text-gray-900">Update Stock</h3>
                <button onclick="closeStockModal()" class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times text-xl"></i>
                </button>
            </div>
            
            <form id="stockForm" method="POST">
                @csrf
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Item</label>
                        <p id="modalItemName" class="text-gray-900 font-medium"></p>
                        <p id="modalCurrentStock" class="text-sm text-gray-500"></p>
                    </div>
                    
                    <div>
                        <label for="type" class="block text-sm font-medium text-gray-700 mb-2">Transaction Type</label>
                        <select name="type" id="type" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                            <option value="in">Stock In</option>
                            <option value="out">Stock Out</option>
                            <option value="adjustment">Adjustment</option>
                        </select>
                    </div>
                    
                    <div>
                        <label for="quantity" class="block text-sm font-medium text-gray-700 mb-2">Quantity</label>
                        <input type="number" name="quantity" id="quantity" min="1" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" required>
                    </div>
                    
                    <div>
                        <label for="reason" class="block text-sm font-medium text-gray-700 mb-2">Reason</label>
                        <input type="text" name="reason" id="reason" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" required>
                    </div>
                    
                    <div>
                        <label for="notes" class="block text-sm font-medium text-gray-700 mb-2">Notes (Optional)</label>
                        <textarea name="notes" id="notes" rows="3" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"></textarea>
                    </div>
                </div>
                
                <div class="flex space-x-3 mt-6">
                    <button type="button" onclick="closeStockModal()" class="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors">
                        Cancel
                    </button>
                    <button type="submit" class="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                        Update Stock
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function showStockModal(itemId, itemName, currentStock) {
    document.getElementById('modalItemName').textContent = itemName;
    document.getElementById('modalCurrentStock').textContent = `Current stock: ${currentStock}`;
    document.getElementById('stockForm').action = `/inventory/items/${itemId}/update-stock`;
    document.getElementById('stockModal').classList.remove('hidden');
}

function closeStockModal() {
    document.getElementById('stockModal').classList.add('hidden');
    document.getElementById('stockForm').reset();
}

// Close modal when clicking outside
document.getElementById('stockModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeStockModal();
    }
});
</script>
@endsection
