@extends('components.app-layout')

@section('title', 'Create Birthday Message')
@section('subtitle', 'Send personalized birthday messages to members')

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="flex flex-col lg:flex-row justify-between items-start lg:items-center gap-4">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Create Birthday Message</h1>
            <p class="text-gray-600">Send personalized birthday messages to selected members</p>
        </div>
        <div class="flex gap-3">
            <a href="{{ route('birthday-messages.index') }}" class="inline-flex items-center px-4 py-2 bg-gray-100 text-gray-700 font-medium rounded-xl hover:bg-gray-200 transition-colors">
                <i class="fas fa-arrow-left mr-2"></i>
                Back to Messages
            </a>
        </div>
    </div>

    <!-- Create Message Form -->
    <div class="bg-white rounded-2xl shadow-lg border border-gray-100 p-6">
        <form method="POST" action="{{ route('birthday-messages.store') }}">
            @csrf
            
            <div class="space-y-6">
                <!-- Member Selection -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-3">Select Members</label>
                    <div class="border border-gray-300 rounded-lg p-4 max-h-64 overflow-y-auto">
                        <div class="mb-4">
                            <label class="flex items-center">
                                <input type="checkbox" id="select-all" class="rounded border-gray-300 text-blue-600 shadow-sm focus:border-blue-300 focus:ring focus:ring-blue-200 focus:ring-opacity-50">
                                <span class="ml-2 text-sm font-medium text-gray-700">Select All Members</span>
                            </label>
                        </div>
                        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                            @foreach($members as $member)
                                <label class="flex items-center p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors">
                                    <input type="checkbox" name="member_ids[]" value="{{ $member->id }}" class="member-checkbox rounded border-gray-300 text-blue-600 shadow-sm focus:border-blue-300 focus:ring focus:ring-blue-200 focus:ring-opacity-50">
                                    <div class="ml-3 flex-1">
                                        <div class="text-sm font-medium text-gray-900">{{ $member->full_name }}</div>
                                        <div class="text-xs text-gray-500">
                                            @if($member->date_of_birth)
                                                Born: {{ $member->date_of_birth->format('M j') }} ({{ $member->age }} years)
                                            @endif
                                        </div>
                                        <div class="text-xs text-gray-500">
                                            @if($member->phone)
                                                <i class="fas fa-phone mr-1"></i>{{ $member->phone }}
                                            @endif
                                            @if($member->email)
                                                <i class="fas fa-envelope mr-1 ml-2"></i>{{ $member->email }}
                                            @endif
                                        </div>
                                    </div>
                                </label>
                            @endforeach
                        </div>
                    </div>
                    @error('member_ids')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>

                <!-- Message Configuration -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label for="message_type" class="block text-sm font-medium text-gray-700 mb-2">Message Type</label>
                        <select name="message_type" id="message_type" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" required>
                            <option value="sms" {{ old('message_type') == 'sms' ? 'selected' : '' }}>SMS Only</option>
                            <option value="email" {{ old('message_type') == 'email' ? 'selected' : '' }}>Email Only</option>
                            <option value="both" {{ old('message_type') == 'both' ? 'selected' : '' }}>Both SMS & Email</option>
                        </select>
                        @error('message_type')
                            <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                        @enderror
                    </div>

                    <div>
                        <label for="send_date" class="block text-sm font-medium text-gray-700 mb-2">Send Date</label>
                        <input type="date" name="send_date" id="send_date" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" value="{{ old('send_date', date('Y-m-d')) }}" required>
                        @error('send_date')
                            <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                        @enderror
                    </div>
                </div>

                <!-- Message Content -->
                <div>
                    <label for="message_content" class="block text-sm font-medium text-gray-700 mb-2">Message Content</label>
                    <textarea name="message_content" id="message_content" rows="6" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500" placeholder="Enter your birthday message here..." required>{{ old('message_content') }}</textarea>
                    <div class="mt-2 text-sm text-gray-500">
                        <p><strong>Available placeholders:</strong></p>
                        <ul class="list-disc list-inside mt-1 space-y-1">
                            <li><code>{name}</code> - Member's first name</li>
                            <li><code>{full_name}</code> - Member's full name</li>
                            <li><code>{age}</code> - Member's age</li>
                            <li><code>{year_group}</code> - Member's year group</li>
                        </ul>
                    </div>
                    @error('message_content')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>

                <!-- Message Templates -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Quick Templates</label>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <button type="button" onclick="useTemplate('template1')" class="p-4 text-left border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-colors">
                            <div class="font-medium text-gray-900 mb-1">Warm Birthday Wishes</div>
                            <div class="text-sm text-gray-600">🎉 Happy Birthday {name}! May God bless you with joy, peace, and prosperity as you celebrate another year of life. Enjoy your special day! - Presbyterian Church of Ghana</div>
                        </button>
                        
                        <button type="button" onclick="useTemplate('template2')" class="p-4 text-left border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-colors">
                            <div class="font-medium text-gray-900 mb-1">Spiritual Birthday Blessing</div>
                            <div class="text-sm text-gray-600">🎂 Wishing you a very Happy Birthday {name}! May this new year of your life be filled with God's abundant blessings and grace. Have a wonderful celebration!</div>
                        </button>
                        
                        <button type="button" onclick="useTemplate('template3')" class="p-4 text-left border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-colors">
                            <div class="font-medium text-gray-900 mb-1">Church Family Wishes</div>
                            <div class="text-sm text-gray-600">🌟 Happy Birthday {name}! On your special day, we pray that God continues to guide your steps and shower you with His love. Celebrate and enjoy! - Your Church Family</div>
                        </button>
                        
                        <button type="button" onclick="useTemplate('template4')" class="p-4 text-left border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-colors">
                            <div class="font-medium text-gray-900 mb-1">Age-Specific Wishes</div>
                            <div class="text-sm text-gray-600">🎈 It's your {age}th birthday {name}! May God grant you health, happiness, and success in the year ahead. Have a blessed and joyful celebration!</div>
                        </button>
                    </div>
                </div>

                <!-- Submit Buttons -->
                <div class="flex flex-col sm:flex-row gap-4 pt-6 border-t border-gray-200">
                    <button type="submit" class="flex-1 inline-flex items-center justify-center px-6 py-3 bg-gradient-to-r from-blue-600 to-blue-700 text-white font-medium rounded-xl hover:from-blue-700 hover:to-blue-800 transition-all duration-200">
                        <i class="fas fa-paper-plane mr-2"></i>
                        Create Birthday Messages
                    </button>
                    <a href="{{ route('birthday-messages.index') }}" class="flex-1 inline-flex items-center justify-center px-6 py-3 bg-gray-100 text-gray-700 font-medium rounded-xl hover:bg-gray-200 transition-colors">
                        <i class="fas fa-times mr-2"></i>
                        Cancel
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<script>
// Select All functionality
document.getElementById('select-all').addEventListener('change', function() {
    const checkboxes = document.querySelectorAll('.member-checkbox');
    checkboxes.forEach(checkbox => {
        checkbox.checked = this.checked;
    });
});

// Update Select All when individual checkboxes change
document.querySelectorAll('.member-checkbox').forEach(checkbox => {
    checkbox.addEventListener('change', function() {
        const allCheckboxes = document.querySelectorAll('.member-checkbox');
        const checkedCheckboxes = document.querySelectorAll('.member-checkbox:checked');
        const selectAllCheckbox = document.getElementById('select-all');
        
        selectAllCheckbox.checked = allCheckboxes.length === checkedCheckboxes.length;
        selectAllCheckbox.indeterminate = checkedCheckboxes.length > 0 && checkedCheckboxes.length < allCheckboxes.length;
    });
});

// Template functions
function useTemplate(templateId) {
    const templates = {
        'template1': '🎉 Happy Birthday {name}! May God bless you with joy, peace, and prosperity as you celebrate another year of life. Enjoy your special day! - Presbyterian Church of Ghana',
        'template2': '🎂 Wishing you a very Happy Birthday {name}! May this new year of your life be filled with God\'s abundant blessings and grace. Have a wonderful celebration!',
        'template3': '🌟 Happy Birthday {name}! On your special day, we pray that God continues to guide your steps and shower you with His love. Celebrate and enjoy! - Your Church Family',
        'template4': '🎈 It\'s your {age}th birthday {name}! May God grant you health, happiness, and success in the year ahead. Have a blessed and joyful celebration!'
    };
    
    document.getElementById('message_content').value = templates[templateId];
}
</script>
@endsection
