<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\Role;
use App\Models\Permission;

class RolesAndPermissionsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Define permissions grouped by module
        $permissions = [
            'members' => [
                ['name' => 'View Members', 'slug' => 'members.view'],
                ['name' => 'Create Members', 'slug' => 'members.create'],
                ['name' => 'Edit Members', 'slug' => 'members.edit'],
                ['name' => 'Delete Members', 'slug' => 'members.delete'],
                ['name' => 'Export Members', 'slug' => 'members.export'],
                ['name' => 'Import Members', 'slug' => 'members.import'],
            ],
            'events' => [
                ['name' => 'View Events', 'slug' => 'events.view'],
                ['name' => 'Create Events', 'slug' => 'events.create'],
                ['name' => 'Edit Events', 'slug' => 'events.edit'],
                ['name' => 'Delete Events', 'slug' => 'events.delete'],
            ],
            'payments' => [
                ['name' => 'View Payments', 'slug' => 'payments.view'],
                ['name' => 'Create Payments', 'slug' => 'payments.create'],
                ['name' => 'Edit Payments', 'slug' => 'payments.edit'],
                ['name' => 'Delete Payments', 'slug' => 'payments.delete'],
                ['name' => 'View Financial Reports', 'slug' => 'payments.reports'],
            ],
            'attendance' => [
                ['name' => 'View Attendance', 'slug' => 'attendance.view'],
                ['name' => 'Record Attendance', 'slug' => 'attendance.create'],
                ['name' => 'Edit Attendance', 'slug' => 'attendance.edit'],
                ['name' => 'Delete Attendance', 'slug' => 'attendance.delete'],
            ],
            'ministries' => [
                ['name' => 'View Ministries', 'slug' => 'ministries.view'],
                ['name' => 'Create Ministries', 'slug' => 'ministries.create'],
                ['name' => 'Edit Ministries', 'slug' => 'ministries.edit'],
                ['name' => 'Delete Ministries', 'slug' => 'ministries.delete'],
            ],
            'users' => [
                ['name' => 'View Users', 'slug' => 'users.view'],
                ['name' => 'Create Users', 'slug' => 'users.create'],
                ['name' => 'Edit Users', 'slug' => 'users.edit'],
                ['name' => 'Delete Users', 'slug' => 'users.delete'],
            ],
            'roles' => [
                ['name' => 'View Roles', 'slug' => 'roles.view'],
                ['name' => 'Create Roles', 'slug' => 'roles.create'],
                ['name' => 'Edit Roles', 'slug' => 'roles.edit'],
                ['name' => 'Delete Roles', 'slug' => 'roles.delete'],
                ['name' => 'Assign Roles', 'slug' => 'roles.assign'],
            ],
            'settings' => [
                ['name' => 'View Settings', 'slug' => 'settings.view'],
                ['name' => 'Edit Settings', 'slug' => 'settings.edit'],
            ],
            'reports' => [
                ['name' => 'View Reports', 'slug' => 'reports.view'],
                ['name' => 'Generate Reports', 'slug' => 'reports.generate'],
                ['name' => 'Export Reports', 'slug' => 'reports.export'],
            ],
            'messages' => [
                ['name' => 'View Messages', 'slug' => 'messages.view'],
                ['name' => 'Send Messages', 'slug' => 'messages.send'],
                ['name' => 'Send Bulk SMS', 'slug' => 'messages.bulk-sms'],
            ],
        ];

        // Create permissions
        $createdPermissions = [];
        foreach ($permissions as $module => $perms) {
            foreach ($perms as $perm) {
                $permission = Permission::create([
                    'name' => $perm['name'],
                    'slug' => $perm['slug'],
                    'module' => $module,
                    'description' => "Permission to {$perm['name']} in {$module} module"
                ]);
                $createdPermissions[$perm['slug']] = $permission->id;
            }
        }

        // Create roles
        $superAdmin = Role::create([
            'name' => 'Super Administrator',
            'slug' => 'super-admin',
            'description' => 'Has full access to all system features',
            'is_active' => true,
        ]);

        $admin = Role::create([
            'name' => 'Administrator',
            'slug' => 'admin',
            'description' => 'Has access to most system features except critical settings',
            'is_active' => true,
        ]);

        $pastor = Role::create([
            'name' => 'Pastor',
            'slug' => 'pastor',
            'description' => 'Can manage members, events, and view reports',
            'is_active' => true,
        ]);

        $secretary = Role::create([
            'name' => 'Secretary',
            'slug' => 'secretary',
            'description' => 'Can manage members, attendance, and basic operations',
            'is_active' => true,
        ]);

        $treasurer = Role::create([
            'name' => 'Treasurer',
            'slug' => 'treasurer',
            'description' => 'Can manage payments and view financial reports',
            'is_active' => true,
        ]);

        // Assign all permissions to Super Admin
        $superAdmin->permissions()->sync(array_values($createdPermissions));

        // Assign permissions to Admin (all except critical settings)
        $adminPermissions = array_filter($createdPermissions, function($key) {
            return !in_array($key, ['roles.delete', 'users.delete']);
        }, ARRAY_FILTER_USE_KEY);
        $admin->permissions()->sync(array_values($adminPermissions));

        // Assign permissions to Pastor
        $pastorPermissions = [
            'members.view', 'members.create', 'members.edit',
            'events.view', 'events.create', 'events.edit',
            'attendance.view', 'attendance.create', 'attendance.edit',
            'ministries.view', 'ministries.create', 'ministries.edit',
            'reports.view', 'reports.generate',
            'messages.view', 'messages.send',
        ];
        $pastor->permissions()->sync(array_intersect_key($createdPermissions, array_flip($pastorPermissions)));

        // Assign permissions to Secretary
        $secretaryPermissions = [
            'members.view', 'members.create', 'members.edit',
            'events.view', 'events.create', 'events.edit',
            'attendance.view', 'attendance.create', 'attendance.edit',
            'messages.view', 'messages.send',
        ];
        $secretary->permissions()->sync(array_intersect_key($createdPermissions, array_flip($secretaryPermissions)));

        // Assign permissions to Treasurer
        $treasurerPermissions = [
            'members.view',
            'payments.view', 'payments.create', 'payments.edit', 'payments.reports',
            'reports.view', 'reports.generate', 'reports.export',
        ];
        $treasurer->permissions()->sync(array_intersect_key($createdPermissions, array_flip($treasurerPermissions)));

        $this->command->info('Roles and permissions seeded successfully!');
    }
}
