<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Hash;
class Member extends Authenticatable
{
    use HasFactory, SoftDeletes, Notifiable;

    protected $fillable = [
        'first_name',
        'last_name',
        'surname',
        'middle_name',
        'title',
        'email',
        'phone',
        'mobile_phone',
        'other_phone',
        'whatsapp_phone',
        'alternate_phone',
        'date_of_birth',
        'gender',
        'marital_status',
        'home_town',
        'aka',
        'address',
        'residential_address',
        'area',
        'city',
        'state',
        'region_state',
        'postal_code',
        'country',
        'nationality',
        'ghana_card_number',
        'profession',
        'occupation',
        'employer',
        'highest_qualification',
        'day_born',
        'currently_schooling',
        'school_name',
        'school_admission_date',
        'school_completion_date',
        'school_telephone',
        'school_location',
        'membership_date',
        'membership_status',
        'membership_type',
        'family_id',
        'year_group_id',
        'relationship_to_head',
        'emergency_contact_name',
        'emergency_contact_phone',
        'emergency_contact_relationship',
        'emergency_contact_occupation',
        'emergency_contact_address',
        'guardian_name',
        'guardian_relationship', 
        'guardian_phone',
        'guardian_address',
        'medical_conditions',
        'special_needs',
        'physically_challenged',
        'disability_type',
        'disability_type_other',
        'photo_path',
        'notes',
        'is_baptized',
        'baptism_date',
        'place_of_baptism',
        'baptism_certificate_number',
        'baptism_type',
        'baptism_officiating_minister',
        'is_confirmed',
        'confirmation_date',
        'place_of_confirmation',
        'confirmation_certificate_number',
        'age_confirmed',
        'confirmation_bible_verse',
        'confirmation_officiating_minister',
        'is_married',
        'marriage_date',
        'place_of_marriage',
        'marriage_certificate_number',
        'marriage_type',
        'marriage_officiating_minister',
        'has_family',
        'skills_talents',
        'interests',
        'receive_newsletter',
        'receive_sms',
        'is_active',
        'password',
        'member_id',
        'credentials_email_sent_at',
        'registration_email_sent_at',
        'last_birthday_email_sent_at',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'membership_date' => 'date',
        'baptism_date' => 'date',
        'confirmation_date' => 'date',
        'marriage_date' => 'date',
        'school_admission_date' => 'date',
        'school_completion_date' => 'date',
        'skills_talents' => 'array',
        'interests' => 'array',
        'is_baptized' => 'boolean',
        'is_confirmed' => 'boolean',
        'is_married' => 'boolean',
        'has_family' => 'boolean',
        'physically_challenged' => 'boolean',
        'currently_schooling' => 'boolean',
        'receive_newsletter' => 'boolean',
        'receive_sms' => 'boolean',
        'is_active' => 'boolean',
        'password' => 'hashed',
        'credentials_email_sent_at' => 'datetime',
        'registration_email_sent_at' => 'datetime',
        'last_birthday_email_sent_at' => 'datetime'
    ];

    // Relationships
    public function family()
    {
        return $this->belongsTo(Family::class);
    }

    public function yearGroup()
    {
        return $this->belongsTo(YearGroup::class);
    }

    public function ministries()
    {
        return $this->belongsToMany(Ministry::class, 'member_ministry')
                    ->withPivot(['role', 'joined_date', 'left_date', 'is_active', 'notes'])
                    ->withTimestamps();
    }

    public function activeMinistries()
    {
        return $this->ministries()->wherePivot('is_active', true);
    }

    public function donations()
    {
        return $this->hasMany(Donation::class);
    }

    public function eventAttendances()
    {
        return $this->hasMany(EventAttendance::class);
    }

    public function organizedEvents()
    {
        return $this->hasMany(Event::class, 'organizer_id');
    }

    public function ledMinistries()
    {
        return $this->hasMany(Ministry::class, 'leader_id');
    }

    public function payments()
    {
        return $this->hasMany(MemberPayment::class, 'member_id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByMembershipStatus($query, $status)
    {
        return $query->where('membership_status', $status);
    }

    public function scopeByMembershipType($query, $type)
    {
        return $query->where('membership_type', $type);
    }

    public function scopeByGender($query, $gender)
    {
        return $query->where('gender', $gender);
    }

    public function scopeByAgeRange($query, $minAge, $maxAge)
    {
        $minDate = now()->subYears($maxAge)->format('Y-m-d');
        $maxDate = now()->subYears($minAge)->format('Y-m-d');
        
        return $query->whereBetween('date_of_birth', [$minDate, $maxDate]);
    }

    // Accessors
    public function getFullNameAttribute()
    {
        $name = $this->first_name;
        
        if ($this->middle_name) {
            $name .= ' ' . $this->middle_name;
        }
        
        $name .= ' ' . $this->last_name;
        
        return $this->title ? $this->title . ' ' . $name : $name;
    }

    public function getAgeAttribute()
    {
        return $this->date_of_birth ? $this->date_of_birth->age : null;
    }

    public function getTotalDonationsAttribute()
    {
        return $this->donations()->sum('amount');
    }

    public function getYearlyDonationsAttribute()
    {
        return $this->donations()
                    ->whereYear('donation_date', now()->year)
                    ->sum('amount');
    }

    // Methods
    public function generateMemberId()
    {
        // Get prefix from settings or use default
        $prefix = $this->getMemberIdPrefix();
        
        // Get last two digits of current year
        $yearSuffix = substr(now()->year, -2);
        
        // Create the year pattern to search for (prefix + year)
        $yearPattern = $prefix . $yearSuffix;
        
        // Find the last member with this year pattern
        $lastMember = static::where('member_id', 'like', $yearPattern . '%')
                           ->orderBy('member_id', 'desc')
                           ->first();
        
        if ($lastMember) {
            // Extract the sequential number from the last member ID
            $sequentialPart = substr($lastMember->member_id, strlen($yearPattern));
            $lastNumber = (int) $sequentialPart;
            $nextNumber = str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $nextNumber = '0001';
        }
        
        return $yearPattern . $nextNumber;
    }

    /**
     * Get the member ID prefix from settings or return default
     */
    private function getMemberIdPrefix()
    {
        try {
            // Try to get prefix from settings table
            $setting = \DB::table('settings')->where('key', 'member_id_prefix')->first();
            return $setting ? $setting->value : 'PCG';
        } catch (\Exception $e) {
            // Fallback to default prefix if settings table doesn't exist or error occurs
            return 'PCG';
        }
    }

    public function isMinor()
    {
        return $this->age && $this->age < 18;
    }

    public function canReceiveCommunication($type = 'email')
    {
        if ($type === 'email') {
            return !empty($this->email) && $this->receive_newsletter;
        } elseif ($type === 'sms') {
            return !empty($this->phone) && $this->receive_sms;
        }
        
        return false;
    }

    /**
     * Automatically assign year group based on member's age
     */
    public function assignYearGroup(): void
    {
        if (!$this->date_of_birth) {
            return;
        }

        $age = $this->age;
        if ($age !== null) {
            $yearGroup = YearGroup::getForAge($age);
            if ($yearGroup) {
                $this->year_group_id = $yearGroup->id;
            }
        }
    }

    /**
     * Get the member's current year group name
     */
    public function getYearGroupNameAttribute(): string
    {
        return $this->yearGroup ? $this->yearGroup->name : 'Not Assigned';
    }

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($member) {
            if (empty($member->member_id)) {
                $member->member_id = $member->generateMemberId();
            }
            
            // Auto-assign year group based on age
            $member->assignYearGroup();
        });
        
        static::updating(function ($member) {
            // Re-assign year group if date of birth changed
            if ($member->isDirty('date_of_birth')) {
                $member->assignYearGroup();
            }
        });
    }
}
