<?php

namespace App\Imports;

use App\Models\Member;
use App\Models\Family;
use App\Models\YearGroup;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithValidation;
use Carbon\Carbon;

class MembersImport implements ToCollection, WithHeadingRow, WithChunkReading
{
    private $results = [
        'success' => 0,
        'errors' => 0,
        'error_details' => []
    ];
    
    private $memberIdCounter;
    private $existingPhones;
    private $existingEmails;
    private $familyCache;

    public function __construct()
    {
        // Cache the starting member ID counter
        $lastMember = Member::orderBy('id', 'desc')->first();
        $this->memberIdCounter = $lastMember ? (intval(substr($lastMember->member_id, 3))) : 0;
        
        // Cache existing phones and emails to avoid repeated queries
        $this->existingPhones = Member::whereNotNull('phone')->pluck('phone')->flip()->toArray();
        $this->existingEmails = Member::whereNotNull('email')->pluck('email')->flip()->toArray();
        
        // Initialize family cache
        $this->familyCache = [];
    }

    public function collection(Collection $rows)
    {
        $membersToInsert = [];
        $familyUpdates = [];
        
        foreach ($rows as $index => $row) {
            try {
                // Skip empty rows
                if ($this->isEmptyRow($row)) {
                    continue;
                }

                // Validate row data
                $validator = $this->validateRow($row->toArray(), $index + 2); // +2 because of header and 0-based index
                
                if ($validator->fails()) {
                    $this->results['errors']++;
                    $this->results['error_details'][] = [
                        'row' => $index + 2,
                        'errors' => $validator->errors()->all()
                    ];
                    continue;
                }

                // Process the member data
                $memberData = $this->prepareMemberData($row);
                
                // Check if member already exists using cached data
                $phoneExists = isset($this->existingPhones[$memberData['phone']]);
                $emailExists = !empty($memberData['email']) && isset($this->existingEmails[$memberData['email']]);

                if ($phoneExists || $emailExists) {
                    $this->results['errors']++;
                    $this->results['error_details'][] = [
                        'row' => $index + 2,
                        'errors' => ['Member with this phone number' . ($emailExists ? ' or email' : '') . ' already exists']
                    ];
                    continue;
                }

                // Add to batch insert array
                $membersToInsert[] = $memberData;
                
                // Add to cache to prevent duplicates within the same import
                $this->existingPhones[$memberData['phone']] = true;
                if (!empty($memberData['email'])) {
                    $this->existingEmails[$memberData['email']] = true;
                }
                
                // Store family info for later processing
                if (!empty($row['family_name'])) {
                    $familyUpdates[] = [
                        'phone' => $memberData['phone'], // We'll use this to find the member later
                        'family_name' => $row['family_name'],
                        'address' => $memberData['address'],
                        'email' => $memberData['email']
                    ];
                }

                $this->results['success']++;

            } catch (\Exception $e) {
                $this->results['errors']++;
                $this->results['error_details'][] = [
                    'row' => $index + 2,
                    'errors' => ['Unexpected error: ' . $e->getMessage()]
                ];
            }
        }
        
        // Batch insert all members
        if (!empty($membersToInsert)) {
            DB::transaction(function () use ($membersToInsert, $familyUpdates) {
                // Insert members in chunks to avoid memory issues
                foreach (array_chunk($membersToInsert, 100) as $chunk) {
                    Member::insert($chunk);
                }
                
                // Process family assignments
                if (!empty($familyUpdates)) {
                    $this->processFamilyAssignments($familyUpdates);
                }
            });
        }
    }

    private function isEmptyRow($row)
    {
        return empty($row['first_name']) && empty($row['last_name']) && empty($row['email']);
    }

    private function validateRow($row, $rowNumber)
    {
        return Validator::make($row, [
            'first_name' => 'required|string|max:255',
            'middle_name' => 'nullable|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'required|string|max:20',
            'date_of_birth' => 'nullable|date',
            'gender' => 'required|in:male,female',
            'membership_type' => 'nullable|in:communicant,baptized,confirmed,long_distance,adherent,catechumen',
            'membership_status' => 'nullable|in:active,inactive,transferred,deceased',
            'marital_status' => 'nullable|in:single,married,divorced,widowed',
            'address' => 'nullable|string|max:500',
            'occupation' => 'nullable|string|max:255',
            'emergency_contact_name' => 'nullable|string|max:255',
            'emergency_contact_phone' => 'nullable|string|max:20',
        ]);
    }

    private function prepareMemberData($row)
    {
        // Generate member ID using cached counter
        $memberIdPrefix = 'PCG';
        $this->memberIdCounter++;
        $memberId = $memberIdPrefix . str_pad($this->memberIdCounter, 4, '0', STR_PAD_LEFT);

        // Calculate day_born from date_of_birth (day name, not day number)
        $dayBorn = null;
        $yearGroupId = null;
        
        if (!empty($row['date_of_birth'])) {
            try {
                $dateOfBirth = Carbon::parse($row['date_of_birth']);
                $dayBorn = $dateOfBirth->format('l'); // 'l' returns full day name (Sunday, Monday, etc.)
                
                // Calculate age and auto-assign year group
                $age = $dateOfBirth->age;
                if ($age >= 0) {
                    $yearGroup = YearGroup::getForAge($age);
                    if ($yearGroup) {
                        $yearGroupId = $yearGroup->id;
                    }
                }
            } catch (\Exception $e) {
                // If date parsing fails, leave day_born and year_group_id as null
            }
        }

        $now = now();
        
        return [
            'member_id' => $memberId,
            'first_name' => $row['first_name'],
            'middle_name' => !empty($row['middle_name']) ? $row['middle_name'] : null,
            'last_name' => $row['last_name'],
            'email' => !empty($row['email']) ? $row['email'] : null,
            'phone' => $row['phone'],
            'date_of_birth' => !empty($row['date_of_birth']) ? Carbon::parse($row['date_of_birth'])->format('Y-m-d') : null,
            'day_born' => $dayBorn,
            'gender' => $row['gender'] ?? 'male',
            'membership_type' => $row['membership_type'] ?? 'communicant',
            'membership_status' => $row['membership_status'] ?? 'active',
            'membership_date' => now()->format('Y-m-d'),
            'marital_status' => $row['marital_status'] ?? 'single',
            'address' => $row['address'] ?? null,
            'residential_address' => $row['address'] ?? null, // Use same address for both
            'occupation' => $row['occupation'] ?? null,
            'profession' => $row['occupation'] ?? null, // Use same value for both
            'emergency_contact_name' => $row['emergency_contact_name'] ?? null,
            'emergency_contact_phone' => $row['emergency_contact_phone'] ?? null,
            'year_group_id' => $yearGroupId,
            'password' => Hash::make('password123'), // Default password
            'is_active' => true,
            'receive_newsletter' => true,
            'receive_sms' => true,
            'created_at' => $now,
            'updated_at' => $now,
        ];
    }

    private function processFamilyAssignments($familyUpdates)
    {
        // Group by family name to reduce queries
        $familiesByName = [];
        foreach ($familyUpdates as $update) {
            $familiesByName[$update['family_name']][] = $update;
        }
        
        foreach ($familiesByName as $familyName => $members) {
            // Get or create family
            if (!isset($this->familyCache[$familyName])) {
                $firstMember = $members[0];
                $family = Family::firstOrCreate(
                    ['family_name' => $familyName],
                    [
                        'address' => $firstMember['address'],
                        'home_phone' => $firstMember['phone'],
                        'email' => $firstMember['email'],
                        'is_active' => true,
                    ]
                );
                $this->familyCache[$familyName] = $family->id;
            }
            
            $familyId = $this->familyCache[$familyName];
            
            // Update all members in this family
            $phones = array_column($members, 'phone');
            Member::whereIn('phone', $phones)->update(['family_id' => $familyId]);
        }
    }
    
    public function chunkSize(): int
    {
        return 100; // Process 100 rows at a time
    }

    public function getResults()
    {
        return $this->results;
    }
}
