<?php

namespace App\Http\Controllers;

use App\Models\YearGroup;
use App\Models\Member;
use Illuminate\Http\Request;

class YearGroupController extends Controller
{
    /**
     * Display a listing of the year groups with member statistics.
     */
    public function index()
    {
        $yearGroups = YearGroup::active()
            ->ordered()
            ->withCount(['members', 'members as active_members_count' => function ($query) {
                $query->where('membership_status', 'active');
            }])
            ->get();

        $stats = [
            'total_groups' => $yearGroups->count(),
            'total_members' => $yearGroups->sum('members_count'),
            'active_members' => $yearGroups->sum('active_members_count'),
            'unassigned_members' => Member::whereNull('year_group_id')->count(),
        ];

        return view('year-groups.index', compact('yearGroups', 'stats'));
    }

    /**
     * Display the specified year group with its members.
     */
    public function show(YearGroup $yearGroup)
    {
        $members = $yearGroup->members()
            ->with(['family'])
            ->orderBy('first_name')
            ->orderBy('last_name')
            ->paginate(20);

        $stats = [
            'total_members' => $yearGroup->members()->count(),
            'active_members' => $yearGroup->members()->where('membership_status', 'active')->count(),
            'male_members' => $yearGroup->members()->where('gender', 'male')->count(),
            'female_members' => $yearGroup->members()->where('gender', 'female')->count(),
        ];

        return view('year-groups.show', compact('yearGroup', 'members', 'stats'));
    }

    /**
     * Reassign all members to appropriate year groups based on their current age
     */
    public function reassignAll()
    {
        $members = Member::whereNotNull('date_of_birth')->get();
        $reassigned = 0;

        foreach ($members as $member) {
            $oldYearGroupId = $member->year_group_id;
            $member->assignYearGroup();
            
            if ($member->year_group_id !== $oldYearGroupId) {
                $member->save();
                $reassigned++;
            }
        }

        return redirect()->route('year-groups.index')
            ->with('success', "Successfully reassigned {$reassigned} members to appropriate year groups.");
    }
}
