<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Display the authenticated user's profile.
     */
    public function profile()
    {
        $user = Auth::user();
        return view('users.profile', compact('user'));
    }

    /**
     * Show the form for editing the authenticated user's profile.
     */
    public function editProfile()
    {
        return view('users.profile-edit');
    }

    /**
     * Update the authenticated user's profile.
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user();
        $rules = [
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique('users')->ignore($user->id),
            ],
            'profile_photo' => 'nullable|image|max:2048',
        ];

        // Handle password change
        if ($request->filled('password')) {
            $rules['current_password'] = 'required|string';
            $rules['password'] = 'required|string|min:8|confirmed';
            
            // Verify current password
            if (!Hash::check($request->current_password, $user->password)) {
                return redirect()->route('profile.edit')
                    ->withErrors(['current_password' => 'The current password is incorrect.'])
                    ->withInput();
            }
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return redirect()->route('profile.edit')
                ->withErrors($validator)
                ->withInput();
        }

        $userData = [
            'name' => $request->input('name'),
            'email' => $request->input('email'),
        ];

        if ($request->filled('password')) {
            $userData['password'] = Hash::make($request->input('password'));
        }

        if ($request->hasFile('profile_photo')) {
            // Delete old profile photo if exists
            if ($user->profile_photo) {
                \Storage::disk('public')->delete($user->profile_photo);
            }
            
            $path = $request->file('profile_photo')->store('profile-photos', 'public');
            $userData['profile_photo'] = $path;
            
            // Debug logging
            \Log::info('Profile photo uploaded', [
                'user_id' => $user->id,
                'file_path' => $path,
                'full_path' => storage_path('app/public/' . $path)
            ]);
        }

        $user->update($userData);

        $message = $request->filled('password') ? 
            'Profile and password updated successfully.' : 
            'Profile updated successfully.';

        return redirect()->route('profile.edit')
            ->with('success', $message);
    }

    /**
     * Reset a user's password.
     */
    public function resetPassword(User $user)
    {
        // Check if the current user is an admin
        if (Auth::user()->role !== 'admin') {
            return redirect()->route('users.index')
                ->with('error', 'You are not authorized to reset passwords.');
        }
        
        return view('users.reset-password', compact('user'));
    }
    
    /**
     * Update the user's password.
     */
    public function updatePassword(Request $request, User $user)
    {
        // Check if the current user is an admin
        if (Auth::user()->role !== 'admin') {
            return redirect()->route('users.index')
                ->with('error', 'You are not authorized to reset passwords.');
        }
        
        $request->validate([
            'password' => 'required|string|min:8|confirmed',
        ]);
        
        $user->update([
            'password' => Hash::make($request->password)
        ]);
        
        return redirect()->route('users.show', $user->id)
            ->with('success', 'Password has been reset successfully.');
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Get statistics
        $stats = [
            'total_users' => User::count(),
            'admin_users' => User::where('role', 'admin')->count(),
            'staff_users' => User::where('role', 'staff')->count(),
            'member_users' => User::where('role', 'student')->count(),
            'users_with_roles' => User::whereNotNull('role_id')->count(),
            'recent_users' => User::where('created_at', '>=', now()->subDays(30))->count(),
        ];
        
        $query = User::with('role')->orderBy('name');
        
        // Apply search filter
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }
        
        // Apply role filter
        if ($request->has('role') && !empty($request->role)) {
            $query->where('role', $request->role);
        }
        
        // Apply role_id filter
        if ($request->has('role_id') && !empty($request->role_id)) {
            $query->where('role_id', $request->role_id);
        }
        
        $users = $query->paginate(20)->withQueryString();
        $roles = \App\Models\Role::where('is_active', true)->orderBy('name')->get();
        
        return view('users.index', compact('users', 'stats', 'roles'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $roles = \App\Models\Role::where('is_active', true)->orderBy('name')->get();
        return view('users.create', compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:admin,staff,student',
            'role_id' => 'nullable|exists:roles,id',
            'profile_photo' => 'nullable|image|max:2048',
        ]);

        if ($validator->fails()) {
            return redirect()->route('users.create')
                ->withErrors($validator)
                ->withInput();
        }
        
        $userData = [
            'name' => $request->input('name'),
            'email' => $request->input('email'),
            'password' => Hash::make($request->input('password')),
            'role' => $request->input('role'),
            'role_id' => $request->input('role_id'),
        ];

        if ($request->hasFile('profile_photo')) {
            $path = $request->file('profile_photo')->store('profile-photos', 'public');
            $userData['profile_photo'] = $path;
        }

        $user = User::create($userData);

        return redirect()->route('users.index')
            ->with('success', 'User created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        // Eager load the assignedRole relationship with permissions
        // Note: Using 'assignedRole' instead of 'role' to avoid conflict with role attribute
        $user->load('assignedRole.permissions');
        
        return view('users.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $roles = \App\Models\Role::where('is_active', true)
            ->withCount('permissions')
            ->orderBy('name')
            ->get();
        return view('users.edit', compact('user', 'roles'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $rules = [
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique('users')->ignore($user->id),
            ],
            'profile_photo' => 'nullable|image|max:2048',
        ];

        // Only validate password if it's provided
        if ($request->filled('password')) {
            $rules['password'] = 'required|string|min:8|confirmed';
        }

        // Only admin can change roles
        if (Auth::user()->role === 'admin') {
            $rules['role'] = 'required|in:admin,staff,student';
            $rules['role_id'] = 'nullable|exists:roles,id';
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return redirect()->route('users.edit', $user->id)
                ->withErrors($validator)
                ->withInput();
        }

        $userData = [
            'name' => $request->input('name'),
            'email' => $request->input('email'),
        ];

        if ($request->filled('password')) {
            $userData['password'] = Hash::make($request->input('password'));
        }

        if (Auth::user()->role === 'admin' && $request->has('role')) {
            $userData['role'] = $request->input('role');
        }

        if (Auth::user()->role === 'admin') {
            // Set role_id to null if empty string, otherwise use the value
            $userData['role_id'] = $request->input('role_id') ?: null;
        }

        if ($request->hasFile('profile_photo')) {
            $path = $request->file('profile_photo')->store('profile-photos', 'public');
            $userData['profile_photo'] = $path;
        }

        $user->update($userData);

        return redirect()->route('users.show', $user->id)
            ->with('success', 'User updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        // Prevent self-deletion
        if (Auth::id() === $user->id) {
            return redirect()->route('users.index')
                ->with('error', 'You cannot delete your own account.');
        }

        $user->delete();

        return redirect()->route('users.index')
            ->with('success', 'User deleted successfully.');
    }

    /**
     * Display the user dashboard with statistics (Permission-Aware).
     */
    public function dashboard()
    {
        $user = Auth::user();
        
        // Initialize stats array
        $stats = [];
        $quickActions = [];
        $recentActivities = [];
        
        // Members Module Statistics (if user has members.view permission)
        if ($user->hasPermission('members.view')) {
            $stats['members'] = [
                'total_members' => \App\Models\Member::count(),
                'active_members' => \App\Models\Member::active()->count(),
                'new_this_month' => \App\Models\Member::whereMonth('created_at', now()->month)
                    ->whereYear('created_at', now()->year)
                    ->count(),
                'pending_approval' => \App\Models\Member::where('membership_status', 'pending')->count(),
            ];
            
            // Add quick actions
            if ($user->hasPermission('members.create')) {
                $quickActions[] = [
                    'title' => 'Add New Member',
                    'description' => 'Register new church member',
                    'route' => 'members.create',
                    'icon' => 'fa-user-plus',
                    'color' => 'blue'
                ];
            }
            
            $quickActions[] = [
                'title' => 'View Members',
                'description' => 'Browse member directory',
                'route' => 'members.index',
                'icon' => 'fa-users',
                'color' => 'green'
            ];
            
            // Recent members activity
            $recentActivities['members'] = \App\Models\Member::with('family')
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get();
        }
        
        // Events Module Statistics (if user has events permission)
        if ($user->hasAnyPermission(['events.view', 'events.create'])) {
            $stats['events'] = [
                'total_events' => \App\Models\Event::count(),
                'upcoming_events' => \App\Models\Event::upcoming()->published()->count(),
                'this_month' => \App\Models\Event::whereMonth('start_datetime', now()->month)
                    ->whereYear('start_datetime', now()->year)
                    ->count(),
            ];
            
            if ($user->hasPermission('events.create')) {
                $quickActions[] = [
                    'title' => 'Create Event',
                    'description' => 'Schedule church activities',
                    'route' => 'events.create',
                    'icon' => 'fa-calendar-plus',
                    'color' => 'orange'
                ];
            }
            
            $quickActions[] = [
                'title' => 'View Events',
                'description' => 'Browse all events',
                'route' => 'events.index',
                'icon' => 'fa-calendar-alt',
                'color' => 'purple'
            ];
            
            // Upcoming events
            $recentActivities['events'] = \App\Models\Event::with(['ministry', 'organizer'])
                ->upcoming()
                ->published()
                ->orderBy('start_datetime')
                ->limit(5)
                ->get();
        }
        
        // Payments/Finance Module (if user has payments permission)
        if ($user->hasAnyPermission(['payments.view', 'payments.create'])) {
            $stats['payments'] = [
                'total_payments' => \App\Models\MemberPayment::where('status', 'confirmed')->count(),
                'this_month_amount' => \App\Models\MemberPayment::where('status', 'confirmed')
                    ->whereYear('payment_date', now()->year)
                    ->whereMonth('payment_date', now()->month)
                    ->sum('amount'),
                'this_year_amount' => \App\Models\MemberPayment::where('status', 'confirmed')
                    ->whereYear('payment_date', now()->year)
                    ->sum('amount'),
            ];
            
            if ($user->hasPermission('payments.create')) {
                $quickActions[] = [
                    'title' => 'Record Payment',
                    'description' => 'Add member payment',
                    'route' => 'member-payments.create',
                    'icon' => 'fa-hand-holding-usd',
                    'color' => 'green'
                ];
            }
            
            $quickActions[] = [
                'title' => 'View Payments',
                'description' => 'Browse payment records',
                'route' => 'member-payments.index',
                'icon' => 'fa-receipt',
                'color' => 'indigo'
            ];
        }
        
        // Attendance Module (if user has attendance permission)
        if ($user->hasAnyPermission(['attendance.view', 'attendance.mark'])) {
            $stats['attendance'] = [
                'total_records' => \App\Models\Attendance::verified()->count(),
                'this_month' => \App\Models\Attendance::verified()
                    ->whereMonth('checked_in_at', now()->month)
                    ->whereYear('checked_in_at', now()->year)
                    ->count(),
            ];
            
            $quickActions[] = [
                'title' => 'View Attendance',
                'description' => 'Check attendance records',
                'route' => 'attendance.index',
                'icon' => 'fa-clipboard-check',
                'color' => 'cyan'
            ];
        }
        
        // Ministries Module
        if ($user->hasAnyPermission(['ministries.view', 'ministries.create'])) {
            $stats['ministries'] = [
                'total_ministries' => \App\Models\Ministry::active()->count(),
                'total_members' => \DB::table('member_ministry')
                    ->where('is_active', true)
                    ->distinct('member_id')
                    ->count('member_id'),
            ];
            
            if ($user->hasPermission('ministries.create')) {
                $quickActions[] = [
                    'title' => 'Create Ministry',
                    'description' => 'Add new ministry',
                    'route' => 'ministries.create.public',
                    'icon' => 'fa-hands-praying',
                    'color' => 'pink'
                ];
            }
        }
        
        // Messages/SMS Module
        if ($user->hasAnyPermission(['messages.view', 'messages.send'])) {
            $stats['messages'] = [
                'total_sent' => \App\Models\SmsMessage::whereIn('status', ['sent', 'delivered'])->count(),
                'pending' => \App\Models\SmsMessage::where('status', 'pending')->count(),
            ];
            
            if ($user->hasPermission('messages.send')) {
                $quickActions[] = [
                    'title' => 'Send SMS',
                    'description' => 'Send bulk messages',
                    'route' => 'sms.create',
                    'icon' => 'fa-sms',
                    'color' => 'yellow'
                ];
            }
        }
        
        // Reports Module
        if ($user->hasPermission('reports.view')) {
            $quickActions[] = [
                'title' => 'View Reports',
                'description' => 'Generate church reports',
                'route' => 'donations.stats',
                'icon' => 'fa-chart-bar',
                'color' => 'teal'
            ];
        }
        
        // Settings Module (Admin only)
        if ($user->hasPermission('settings.view')) {
            $quickActions[] = [
                'title' => 'System Settings',
                'description' => 'Configure system',
                'route' => 'settings.index',
                'icon' => 'fa-cog',
                'color' => 'gray'
            ];
        }
        
        // User's role information
        // Load assignedRole if not already loaded
        if ($user->role_id && !$user->relationLoaded('assignedRole')) {
            $user->load('assignedRole');
        }
        
        $roleInfo = [
            'name' => $user->role_id && $user->assignedRole ? $user->assignedRole->name : ucfirst($user->role ?? 'Staff'),
            'legacy_role' => $user->role,
            'permissions_count' => $user->role_id && $user->assignedRole ? $user->assignedRole->permissions()->count() : 0,
        ];
        
        return view('users.dashboard', compact('stats', 'quickActions', 'recentActivities', 'roleInfo'));
    }

    /**
     * Display the user portal with documents.
     */
    public function portal(Request $request)
    {
        $user = Auth::user();
        
        // Get documents with filters
        $query = \App\Models\Document::with(['category', 'uploader'])
            ->where('is_active', true);
            
        // Apply search filter
        if (request('search')) {
            $search = request('search');
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('file_name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }
        
        // Apply category filter
        if (request('category')) {
            $query->where('document_category_id', request('category'));
        }
        
        // Apply file type filter
        if (request('file_type')) {
            $query->where('file_type', request('file_type'));
        }
        
        // Get documents with pagination - sorted alphabetically by title
        $documents = $query->orderBy('title', 'asc')->paginate(12);
        
        // Get categories for filter dropdown
        $categories = \App\Models\DocumentCategory::where('is_active', true)
            ->orderBy('name')
            ->get();
            
        // Get file types for filter
        $fileTypes = \App\Models\Document::select('file_type')
            ->distinct()
            ->whereNotNull('file_type')
            ->orderBy('file_type')
            ->pluck('file_type');
            
        // Get user-specific statistics
        $userStats = [
            'total_prints' => $user->total_prints,
            'unique_documents_printed' => $user->unique_printed_documents,
            'last_print' => $user->documentPrints()->latest('last_printed_at')->first()?->last_printed_at
        ];
            
        // Get general statistics
        $stats = [
            'total_documents' => \App\Models\Document::where('is_active', true)->count(),
            'active_documents' => \App\Models\Document::where('is_active', true)->count(),
            'total_prints' => \App\Models\Document::sum('print_count'),
            'categories_count' => \App\Models\DocumentCategory::where('is_active', true)->count()
        ];
        
        return view('users.portal', compact('documents', 'categories', 'fileTypes', 'stats', 'userStats'));
    }

    /**
     * Get documents for portal (AJAX endpoint).
     */
    public function portalDocuments(Request $request)
    {
        $query = \App\Models\Document::with(['category', 'uploader'])
            ->where('is_active', true)
            ->orderBy('created_at', 'desc');
        
        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('file_name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }
        
        if ($request->filled('category')) {
            $query->where('document_category_id', $request->category);
        }
        
        if ($request->filled('file_type')) {
            $fileType = $request->file_type;
            if ($fileType === 'image') {
                $query->where(function($q) {
                    $q->where('file_name', 'like', '%.jpg')
                      ->orWhere('file_name', 'like', '%.jpeg')
                      ->orWhere('file_name', 'like', '%.png')
                      ->orWhere('file_name', 'like', '%.gif')
                      ->orWhere('file_name', 'like', '%.webp');
                });
            } else {
                $query->where('file_name', 'like', "%.{$fileType}");
            }
        }
        
        $documents = $query->paginate(12)->withQueryString();
        
        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'documents' => $documents->items(),
                'pagination' => [
                    'current_page' => $documents->currentPage(),
                    'last_page' => $documents->lastPage(),
                    'per_page' => $documents->perPage(),
                    'total' => $documents->total()
                ]
            ]);
        }
        
        return redirect()->route('users.portal');
    }

    /**
     * Show bulk upload form for users
     */
    public function bulkUploadForm(Request $request)
    {
        $categoryId = $request->get('category');
        $selectedCategory = $categoryId ? \App\Models\DocumentCategory::findOrFail($categoryId) : null;
        $categories = \App\Models\DocumentCategory::active()->ordered()->get();
        
        return view('users.bulk-upload', compact('categories', 'selectedCategory'));
    }

    /**
     * Store bulk uploaded documents from users
     */
    public function bulkUploadStore(Request $request)
    {
        $validated = $request->validate([
            'document_category_id' => 'required|exists:document_categories,id',
            'files' => 'required|array|min:1',
            'files.*' => 'file|mimes:pdf,doc,docx,txt,jpg,jpeg,png|max:10240', // 10MB max per file
        ]);

        $uploadedFiles = [];
        $errors = [];
        $successCount = 0;
        $totalFiles = count($request->file('files'));

        foreach ($request->file('files') as $index => $file) {
            try {
                // Generate unique filename
                $fileName = time() . '_' . $index . '_' . \Illuminate\Support\Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME)) . '.' . $file->getClientOriginalExtension();
                $filePath = $file->storeAs('documents', $fileName, 'public');

                // Create document record
                $document = \App\Models\Document::create([
                    'title' => pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME),
                    'description' => 'User bulk uploaded document',
                    'document_category_id' => $validated['document_category_id'],
                    'file_name' => $file->getClientOriginalName(),
                    'file_path' => $filePath,
                    'file_type' => $file->getClientMimeType(),
                    'file_size' => $file->getSize(),
                    'uploaded_by' => Auth::id(),
                    'is_printable' => true, // Default to printable for user uploads
                    'is_active' => false // Require admin approval
                ]);

                $uploadedFiles[] = [
                    'name' => $file->getClientOriginalName(),
                    'id' => $document->id,
                    'status' => 'success'
                ];
                $successCount++;

            } catch (\Exception $e) {
                $errors[] = [
                    'name' => $file->getClientOriginalName(),
                    'error' => $e->getMessage(),
                    'status' => 'error'
                ];
            }
        }

        // Return JSON response for AJAX requests
        if ($request->expectsJson()) {
            return response()->json([
                'success' => $successCount > 0,
                'message' => "Successfully uploaded {$successCount} of {$totalFiles} documents. Documents are pending admin approval.",
                'uploaded_files' => $uploadedFiles,
                'errors' => $errors,
                'success_count' => $successCount,
                'total_count' => $totalFiles
            ]);
        }

        // Regular form submission
        $message = "Successfully uploaded {$successCount} of {$totalFiles} documents. Documents are pending admin approval.";
        if (count($errors) > 0) {
            $message .= " " . count($errors) . " files failed to upload.";
        }

        return redirect()->route('users.portal')
            ->with('success', $message)
            ->with('bulk_upload_results', [
                'uploaded_files' => $uploadedFiles,
                'errors' => $errors
            ]);
    }
}
