<?php

namespace App\Http\Controllers;

use App\Models\Member;
use App\Models\MemberPayment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class TithePaymentController extends Controller
{
    /**
     * Display tithe payments grouped by year and month
     */
    public function index(Request $request)
    {
        $year = $request->get('year', now()->year);
        $month = $request->get('month');
        $memberId = $request->get('member_id');

        $query = MemberPayment::where('payment_type', 'tithe')
            ->where('status', 'confirmed')
            ->with(['member', 'recordedBy']);

        // Filter by year
        $query->whereYear('payment_date', $year);

        // Filter by month if provided
        if ($month) {
            $query->whereMonth('payment_date', $month);
        }

        // Filter by member if provided
        if ($memberId) {
            $query->where('member_id', $memberId);
        }

        // Get payments
        $payments = $query->orderBy('payment_date', 'desc')->get();

        // Group by year and month
        $groupedPayments = $payments->groupBy(function($payment) {
            return $payment->payment_date->format('Y-m');
        })->map(function($monthPayments, $yearMonth) {
            return [
                'year_month' => $yearMonth,
                'month_name' => Carbon::createFromFormat('Y-m', $yearMonth)->format('F Y'),
                'total_amount' => $monthPayments->sum('amount'),
                'payment_count' => $monthPayments->count(),
                'members_count' => $monthPayments->pluck('member_id')->unique()->count(),
                'payments' => $monthPayments->groupBy('member_id')->map(function($memberPayments) {
                    return [
                        'member' => $memberPayments->first()->member,
                        'payments' => $memberPayments,
                        'total' => $memberPayments->sum('amount'),
                        'count' => $memberPayments->count()
                    ];
                })
            ];
        });

        // Statistics
        $stats = [
            'total_amount' => $payments->sum('amount'),
            'total_payments' => $payments->count(),
            'unique_members' => $payments->pluck('member_id')->unique()->count(),
            'average_payment' => $payments->count() > 0 ? $payments->avg('amount') : 0,
        ];

        // Available years
        $availableYears = MemberPayment::where('payment_type', 'tithe')
            ->selectRaw('YEAR(payment_date) as year')
            ->distinct()
            ->orderBy('year', 'desc')
            ->pluck('year');

        // Active members for filter
        $members = Member::active()->orderBy('first_name')->get();

        return view('tithes.index', compact('groupedPayments', 'stats', 'year', 'month', 'availableYears', 'members', 'memberId'));
    }

    /**
     * Show individual member's tithe history
     */
    public function memberHistory(Member $member, Request $request)
    {
        $year = $request->get('year', now()->year);

        $payments = MemberPayment::where('member_id', $member->id)
            ->where('payment_type', 'tithe')
            ->where('status', 'confirmed')
            ->whereYear('payment_date', $year)
            ->orderBy('payment_date', 'desc')
            ->get();

        // Group by month
        $monthlyPayments = $payments->groupBy(function($payment) {
            return $payment->payment_date->format('Y-m');
        })->map(function($monthPayments, $yearMonth) {
            return [
                'year_month' => $yearMonth,
                'month_name' => Carbon::createFromFormat('Y-m', $yearMonth)->format('F Y'),
                'payments' => $monthPayments,
                'total' => $monthPayments->sum('amount'),
                'count' => $monthPayments->count()
            ];
        });

        // Yearly statistics
        $yearlyStats = [
            'total_amount' => $payments->sum('amount'),
            'total_payments' => $payments->count(),
            'average_payment' => $payments->count() > 0 ? $payments->avg('amount') : 0,
            'months_paid' => $payments->pluck('payment_date')->map(function($date) {
                return $date->format('Y-m');
            })->unique()->count()
        ];

        // Available years for this member
        $availableYears = MemberPayment::where('member_id', $member->id)
            ->where('payment_type', 'tithe')
            ->selectRaw('YEAR(payment_date) as year')
            ->distinct()
            ->orderBy('year', 'desc')
            ->pluck('year');

        return view('tithes.member-history', compact('member', 'monthlyPayments', 'yearlyStats', 'year', 'availableYears'));
    }

    /**
     * Export tithe report
     */
    public function export(Request $request)
    {
        $year = $request->get('year', now()->year);
        $month = $request->get('month');

        $query = MemberPayment::where('payment_type', 'tithe')
            ->where('status', 'confirmed')
            ->with(['member'])
            ->whereYear('payment_date', $year);

        if ($month) {
            $query->whereMonth('payment_date', $month);
        }

        $payments = $query->orderBy('payment_date', 'desc')->get();

        $filename = 'tithe-report-' . $year . ($month ? '-' . str_pad($month, 2, '0', STR_PAD_LEFT) : '') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($payments) {
            $file = fopen('php://output', 'w');
            
            // CSV headers
            fputcsv($file, [
                'Payment Date',
                'Year-Month',
                'Member ID',
                'Member Name',
                'Amount',
                'Payment Method',
                'Receipt Number',
                'Payment Reference'
            ]);

            // CSV data
            foreach ($payments as $payment) {
                fputcsv($file, [
                    $payment->payment_date->format('Y-m-d'),
                    $payment->payment_date->format('Y-m'),
                    $payment->member->member_id,
                    $payment->member->full_name,
                    $payment->amount,
                    ucfirst(str_replace('_', ' ', $payment->payment_method)),
                    $payment->receipt_number,
                    $payment->payment_reference
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Get tithe statistics
     */
    public function statistics(Request $request)
    {
        $year = $request->get('year', now()->year);

        // Monthly breakdown
        $monthlyStats = MemberPayment::where('payment_type', 'tithe')
            ->where('status', 'confirmed')
            ->whereYear('payment_date', $year)
            ->selectRaw('MONTH(payment_date) as month, COUNT(*) as count, SUM(amount) as total, COUNT(DISTINCT member_id) as unique_members')
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->mapWithKeys(function($stat) use ($year) {
                return [
                    Carbon::createFromDate($year, $stat->month, 1)->format('F') => [
                        'count' => $stat->count,
                        'total' => $stat->total,
                        'unique_members' => $stat->unique_members,
                        'average' => $stat->count > 0 ? $stat->total / $stat->count : 0
                    ]
                ];
            });

        // Top contributors
        $topContributors = MemberPayment::where('payment_type', 'tithe')
            ->where('status', 'confirmed')
            ->whereYear('payment_date', $year)
            ->select('member_id', DB::raw('SUM(amount) as total, COUNT(*) as count'))
            ->groupBy('member_id')
            ->orderBy('total', 'desc')
            ->limit(10)
            ->with('member')
            ->get();

        return response()->json([
            'year' => $year,
            'monthly_stats' => $monthlyStats,
            'top_contributors' => $topContributors
        ]);
    }
}
