<?php

namespace App\Http\Controllers;

use App\Models\Member;
use App\Models\Family;
use App\Models\Ministry;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\MembersExport;
use App\Mail\MemberRegistrationEmail;
use App\Mail\MemberLoginCredentialsEmail;
use App\Mail\MemberWelcomeEmail;

class MemberController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Member::with(['family', 'ministries']);
        
        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('middle_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('member_id', 'like', "%{$search}%");
            });
        }
        
        // Filter by membership status
        if ($request->filled('status')) {
            $query->where('membership_status', $request->status);
        }
        
        // Filter by membership type
        if ($request->filled('type')) {
            $query->where('membership_type', $request->type);
        }
        // Filter by gender
        if ($request->filled('gender')) {
            $query->where('gender', $request->gender);
        }
        
        // Filter by day born
        if ($request->filled('day_born')) {
            $query->where('day_born', $request->day_born);
        }
        
        // Filter by marital status
        if ($request->filled('marital_status')) {
            $query->where('marital_status', $request->marital_status);
        }
        
        // Filter by highest qualification
        if ($request->filled('highest_qualification')) {
            $query->where('highest_qualification', $request->highest_qualification);
        }
        
        // Filter by family
        if ($request->filled('family_id')) {
            $query->where('family_id', $request->family_id);
        }
        
        // Filter by ministry
        if ($request->filled('ministry_id')) {
            $query->whereHas('ministries', function ($q) use ($request) {
                $q->where('ministries.id', $request->ministry_id);
            });
        }
        
        // Filter by year group
        if ($request->filled('year_group_id')) {
            $query->where('year_group_id', $request->year_group_id);
        }
        
        // Sorting
        $sortBy = $request->get('sort', 'first_name');
        $sortDirection = $request->get('direction', 'asc');
        
        if ($sortBy === 'name') {
            $query->orderBy('first_name', $sortDirection)->orderBy('last_name', $sortDirection);
        } elseif ($sortBy === 'membership_date') {
            $query->orderBy('membership_date', $sortDirection);
        } elseif ($sortBy === 'family') {
            $query->join('families', 'members.family_id', '=', 'families.id')
                  ->orderBy('families.family_name', $sortDirection)
                  ->select('members.*');
        } else {
            $query->orderBy($sortBy, $sortDirection);
        }
        
        $members = $query->with(['family', 'yearGroup'])->paginate(20)->withQueryString();
        
        // Calculate statistics
        $stats = [
            'total_members' => Member::count(),
            'active_members' => Member::where('membership_status', 'active')->count(),
            'new_members' => Member::whereMonth('created_at', now()->month)
                                  ->whereYear('created_at', now()->year)
                                  ->count(),
            'pending_members' => Member::where('membership_status', 'pending')->count(),
            'total_families' => Family::count(),
            'active_ministries' => Ministry::where('is_active', true)->count(),
        ];
        
        $families = Family::orderBy('family_name')->get();
        $ministries = Ministry::where('is_active', true)->orderBy('name')->get();
        
        // Get organization settings for dynamic content
        $organizationSettings = [
            'name' => Setting::getValue('organization_name', 'general', 'Presbyterian Church of Ghana'),
            'slogan' => Setting::getValue('organization_slogan', 'general', 'Church Management System'),
            'logo' => Setting::getValue('organization_logo', 'general')
        ];
        
        return view('members.index', compact('members', 'families', 'ministries', 'stats', 'organizationSettings'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        try {
            $families = \App\Models\Family::where('is_active', true)->orderBy('family_name')->get();
        } catch (\Exception $e) {
            $families = collect();
        }
        
        try {
            $ministries = \App\Models\Ministry::where('is_active', true)->orderBy('name')->get();
        } catch (\Exception $e) {
            $ministries = collect();
        }
        
        return view('members.create', compact('families', 'ministries'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'middle_name' => 'nullable|string|max:255',
            'title' => 'nullable|string|max:50',
            'email' => 'nullable|email|unique:members,email',
            'phone' => 'nullable|string|max:20',
            'whatsapp_phone' => 'nullable|string|max:20',
            'alternate_phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date|before:today',
            'gender' => 'nullable|in:male,female',
            'marital_status' => 'nullable|in:single,married,divorced,widowed',
            'home_town' => 'nullable|string|max:255',
            'aka' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:255',
            'state' => 'nullable|string|max:255',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:255',
            'profession' => 'nullable|string|max:255',
            'occupation' => 'nullable|string|max:255',
            'employer' => 'nullable|string|max:255',
            'highest_qualification' => 'nullable|in:none,primary,jhs,shs,tertiary,postgraduate,professional,other',
            'day_born' => 'nullable|in:Sunday,Monday,Tuesday,Wednesday,Thursday,Friday,Saturday',
            'currently_schooling' => 'boolean',
            'school_name' => 'nullable|string|max:255',
            'school_admission_date' => 'nullable|date',
            'school_completion_date' => 'nullable|date|after_or_equal:school_admission_date',
            'school_telephone' => 'nullable|string|max:20',
            'school_location' => 'nullable|string|max:500',
            'membership_date' => 'nullable|date',
            'membership_status' => 'required|in:active,inactive,transferred,deceased,pending',
            'membership_type' => 'nullable|in:communicant,baptized,confirmed,long_distance,adherent,catechumen',
            'family_id' => 'nullable|exists:families,id',
            'relationship_to_head' => 'nullable|string|max:50',
            'emergency_contact_name' => 'nullable|string',
            'emergency_contact_phone' => 'nullable|string|max:20',
            'emergency_contact_relationship' => 'nullable|string|max:100',
            'guardian_name' => 'nullable|string|max:255',
            'guardian_relationship' => 'nullable|string|max:100',
            'guardian_phone' => 'nullable|string|max:20',
            'guardian_address' => 'nullable|string|max:500',
            'medical_conditions' => 'nullable|string',
            'special_needs' => 'nullable|string',
            'physically_challenged' => 'boolean',
            'disability_type' => 'nullable|in:mobility,visual,hearing,speech,intellectual,learning,mental_health,chronic_illness,multiple,other',
            'notes' => 'nullable|string',
            'is_baptized' => 'boolean',
            'baptism_date' => 'nullable|date',
            'place_of_baptism' => 'nullable|string|max:255',
            'baptism_type' => 'nullable|in:sprinkling,immersion,pouring,water,other',
            'baptism_officiating_minister' => 'nullable|string|max:255',
            'is_confirmed' => 'boolean',
            'confirmation_date' => 'nullable|date',
            'place_of_confirmation' => 'nullable|string|max:255',
            'age_confirmed' => 'nullable|integer|min:1|max:100',
            'confirmation_bible_verse' => 'nullable|string|max:500',
            'confirmation_officiating_minister' => 'nullable|string|max:255',
            'is_married' => 'boolean',
            'marriage_date' => 'nullable|date',
            'place_of_marriage' => 'nullable|string|max:255',
            'marriage_type' => 'nullable|in:traditional,customary,ordinance,other',
            'marriage_officiating_minister' => 'nullable|string|max:255',
            'has_family' => 'boolean',
            'skills_talents' => 'nullable|string',
            'interests' => 'nullable|string',
            'receive_newsletter' => 'boolean',
            'receive_sms' => 'boolean',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'ministries' => 'nullable|array',
            'ministries.*' => 'exists:ministries,id'
        ]);
        
        // Handle photo upload
        if ($request->hasFile('photo')) {
            $validated['photo_path'] = $request->file('photo')->store('members/photos', 'public');
        }
        
        $member = Member::create($validated);
        
        // Attach ministries if provided
        if ($request->filled('ministries')) {
            foreach ($request->ministries as $ministryId) {
                $member->ministries()->attach($ministryId, [
                    'role' => 'member',
                    'joined_date' => now(),
                    'is_active' => true
                ]);
            }
        }
        
        // Send registration email
        $this->sendMemberRegistrationEmail($member);
        
        return redirect()->route('members.show', $member)
                        ->with('success', 'Member created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Member $member)
    {
        $member->load([
            'family',
            'yearGroup',
            'ministries' => function ($query) {
                $query->withPivot(['role', 'joined_date', 'left_date', 'is_active', 'notes']);
            },
            'donations' => function ($query) {
                $query->orderBy('donation_date', 'desc')->limit(10);
            },
            'eventAttendances.event',
            'organizedEvents' => function ($query) {
                $query->upcoming()->limit(5);
            }
        ]);
        
        // Load member payments with latest first
        $payments = $member->payments()
            ->with('recordedBy')
            ->orderBy('payment_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Calculate payment statistics
        $paymentStats = [
            'total_payments' => $payments->count(),
            'total_amount' => $payments->sum('amount'),
            'tithe_payments' => $payments->where('payment_type', 'tithe')->count(),
            'tithe_amount' => $payments->where('payment_type', 'tithe')->sum('amount'),
            'this_year_total' => $payments->where('payment_date', '>=', now()->startOfYear())->sum('amount'),
            'this_month_total' => $payments->where('payment_date', '>=', now()->startOfMonth())->sum('amount'),
        ];
        
        return view('members.show', compact('member', 'payments', 'paymentStats'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Member $member)
    {
        $families = Family::active()->orderBy('family_name')->get();
        $ministries = Ministry::active()->orderBy('name')->get();
        $memberMinistries = $member->ministries->pluck('id')->toArray();
        
        return view('members.edit', compact('member', 'families', 'ministries', 'memberMinistries'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Member $member)
    {
        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'middle_name' => 'nullable|string|max:255',
            'title' => 'nullable|string|max:50',
            'email' => ['nullable', 'email', Rule::unique('members')->ignore($member->id)],
            'phone' => 'nullable|string|max:20',
            'whatsapp_phone' => 'nullable|string|max:20',
            'alternate_phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date|before:today',
            'gender' => 'nullable|in:male,female',
            'marital_status' => 'nullable|in:single,married,divorced,widowed',
            'home_town' => 'nullable|string|max:255',
            'aka' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:255',
            'state' => 'nullable|string|max:255',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:255',
            'profession' => 'nullable|string|max:255',
            'occupation' => 'nullable|string|max:255',
            'employer' => 'nullable|string|max:255',
            'highest_qualification' => 'nullable|in:none,primary,jhs,shs,tertiary,postgraduate,professional,other',
            'day_born' => 'nullable|in:Sunday,Monday,Tuesday,Wednesday,Thursday,Friday,Saturday',
            'currently_schooling' => 'boolean',
            'school_name' => 'nullable|string|max:255',
            'school_admission_date' => 'nullable|date',
            'school_completion_date' => 'nullable|date|after_or_equal:school_admission_date',
            'school_telephone' => 'nullable|string|max:20',
            'school_location' => 'nullable|string|max:500',
            'membership_date' => 'nullable|date',
            'membership_status' => 'required|in:active,inactive,transferred,deceased,pending',
            'membership_type' => 'nullable|in:communicant,baptized,confirmed,long_distance,adherent,catechumen',
            'family_id' => 'nullable|exists:families,id',
            'relationship_to_head' => 'nullable|string|max:50',
            'emergency_contact_name' => 'nullable|string',
            'emergency_contact_phone' => 'nullable|string|max:20',
            'emergency_contact_relationship' => 'nullable|string|max:100',
            'guardian_name' => 'nullable|string|max:255',
            'guardian_relationship' => 'nullable|string|max:100',
            'guardian_phone' => 'nullable|string|max:20',
            'guardian_address' => 'nullable|string|max:500',
            'medical_conditions' => 'nullable|string',
            'special_needs' => 'nullable|string',
            'physically_challenged' => 'boolean',
            'disability_type' => 'nullable|in:mobility,visual,hearing,speech,intellectual,learning,mental_health,chronic_illness,multiple,other',
            'notes' => 'nullable|string',
            'is_baptized' => 'boolean',
            'baptism_date' => 'nullable|date',
            'place_of_baptism' => 'nullable|string|max:255',
            'baptism_type' => 'nullable|in:sprinkling,immersion,pouring,water,other',
            'baptism_officiating_minister' => 'nullable|string|max:255',
            'is_confirmed' => 'boolean',
            'confirmation_date' => 'nullable|date',
            'place_of_confirmation' => 'nullable|string|max:255',
            'age_confirmed' => 'nullable|integer|min:1|max:100',
            'confirmation_bible_verse' => 'nullable|string|max:500',
            'confirmation_officiating_minister' => 'nullable|string|max:255',
            'is_married' => 'boolean',
            'marriage_date' => 'nullable|date',
            'place_of_marriage' => 'nullable|string|max:255',
            'marriage_type' => 'nullable|in:traditional,customary,ordinance,other',
            'marriage_officiating_minister' => 'nullable|string|max:255',
            'has_family' => 'boolean',
            'skills_talents' => 'nullable|string',
            'interests' => 'nullable|string',
            'receive_newsletter' => 'boolean',
            'receive_sms' => 'boolean',
            'is_active' => 'boolean',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'ministries' => 'nullable|array',
            'ministries.*' => 'exists:ministries,id'
        ]);
        
        // Set default membership_type if not provided
        if (empty($validated['membership_type'])) {
            $validated['membership_type'] = 'communicant';
        }
        
        // Handle photo upload
        if ($request->hasFile('photo')) {
            // Delete old photo if exists
            if ($member->photo_path) {
                Storage::disk('public')->delete($member->photo_path);
            }
            $validated['photo_path'] = $request->file('photo')->store('members/photos', 'public');
        }
        
        $member->update($validated);
        
        // Sync ministries
        if ($request->has('ministries')) {
            $member->ministries()->detach();
            foreach ($request->ministries as $ministryId) {
                $member->ministries()->attach($ministryId, [
                    'role' => 'member',
                    'joined_date' => now(),
                    'is_active' => true
                ]);
            }
        }
        
        return redirect()->route('members.show', $member)
                        ->with('success', 'Member updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Member $member)
    {
        // Delete photo if exists
        if ($member->photo_path) {
            Storage::disk('public')->delete($member->photo_path);
        }
        
        $member->delete();
        
        return redirect()->route('members.index')
                        ->with('success', 'Member deleted successfully.');
    }

    /**
     * Bulk delete members
     */
    public function bulkDelete(Request $request)
    {
        $request->validate([
            'member_ids' => 'required|array|min:1',
            'member_ids.*' => 'exists:members,id'
        ]);

        $memberIds = $request->member_ids;
        
        // Get members to delete
        $members = Member::whereIn('id', $memberIds)->get();
        
        $deletedCount = 0;
        foreach ($members as $member) {
            // Delete photo if exists
            if ($member->photo_path) {
                Storage::disk('public')->delete($member->photo_path);
            }
            
            $member->delete();
            $deletedCount++;
        }
        
        return redirect()->route('members.index')
                        ->with('success', "{$deletedCount} member(s) deleted successfully.");
    }

    /**
     * Export members data in multiple formats
     */
    public function export(Request $request)
    {
        $format = $request->get('format', 'csv'); // Default to CSV
        
        $query = Member::with(['family', 'ministries']);
        
        // Apply same filters as index method
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('middle_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('member_id', 'like', "%{$search}%");
            });
        }
        
        if ($request->filled('status')) {
            $query->where('membership_status', $request->status);
        }
        
        if ($request->filled('type')) {
            $query->where('membership_type', $request->type);
        }
        
        if ($request->filled('gender')) {
            $query->where('gender', $request->gender);
        }
        
        // Filter by day born
        if ($request->filled('day_born')) {
            $query->where('day_born', $request->day_born);
        }
        
        // Filter by marital status
        if ($request->filled('marital_status')) {
            $query->where('marital_status', $request->marital_status);
        }
        
        // Filter by highest qualification
        if ($request->filled('highest_qualification')) {
            $query->where('highest_qualification', $request->highest_qualification);
        }
        
        if ($request->filled('family_id')) {
            $query->where('family_id', $request->family_id);
        }
        
        if ($request->filled('ministry_id')) {
            $query->whereHas('ministries', function ($q) use ($request) {
                $q->where('ministries.id', $request->ministry_id);
            });
        }
        
        if ($request->filled('year_group_id')) {
            $query->where('year_group_id', $request->year_group_id);
        }

        $members = $query->orderBy('first_name')->orderBy('last_name')->get();
        
        $timestamp = now()->format('Y-m-d_H-i-s');
        
        switch ($format) {
            case 'excel':
                return Excel::download(new MembersExport($members), "members_export_{$timestamp}.xlsx");
                
            case 'pdf':
                $pdf = Pdf::loadView('exports.members-pdf', compact('members'))
                    ->setPaper('a4', 'landscape')
                    ->setOptions([
                        'defaultFont' => 'sans-serif',
                        'isRemoteEnabled' => true,
                        'isHtml5ParserEnabled' => true,
                    ]);
                return $pdf->download("members_export_{$timestamp}.pdf");
                
            case 'csv':
            default:
                $filename = "members_export_{$timestamp}.csv";
                
                $headers = [
                    'Content-Type' => 'text/csv',
                    'Content-Disposition' => 'attachment; filename="' . $filename . '"',
                ];
                
                $callback = function() use ($members) {
                    $file = fopen('php://output', 'w');
                    
                    // CSV Headers
                    fputcsv($file, [
                        'Member ID',
                        'First Name',
                        'Last Name',
                        'Email',
                        'Phone',
                        'Gender',
                        'Date of Birth',
                        'Age',
                        'Address',
                        'City',
                        'State',
                        'Postal Code',
                        'Family Name',
                        'Membership Status',
                        'Membership Type',
                        'Membership Date',
                        'Ministries',
                        'Created At'
                    ]);
                    
                    // CSV Data
                    foreach ($members as $member) {
                        $age = $member->date_of_birth ? \Carbon\Carbon::parse($member->date_of_birth)->age : '';
                        $ministries = $member->ministries->pluck('name')->join(', ');
                        
                        fputcsv($file, [
                            $member->member_id,
                            $member->first_name,
                            $member->last_name,
                            $member->email,
                            $member->phone,
                            ucfirst($member->gender ?? ''),
                            $member->date_of_birth ? \Carbon\Carbon::parse($member->date_of_birth)->format('Y-m-d') : '',
                            $age,
                            $member->address,
                            $member->city,
                            $member->state,
                            $member->postal_code,
                            $member->family ? $member->family->family_name : '',
                            ucfirst($member->membership_status),
                            ucfirst($member->membership_type),
                            $member->membership_date ? \Carbon\Carbon::parse($member->membership_date)->format('Y-m-d') : '',
                            $ministries,
                            $member->created_at->format('Y-m-d H:i:s')
                        ]);
                    }
                    
                    fclose($file);
                };
                
                return response()->stream($callback, 200, $headers);
        }
    }

    /**
     * Generate member ID card
     */
    public function generateIdCard(Member $member)
    {
        // This would generate a PDF ID card
        // For now, return a simple response
        return response()->json(['message' => 'ID card generation to be implemented']);
    }

    /**
     * Get member statistics
     */
    public function statistics()
    {
        $stats = [
            'total_members' => Member::count(),
            'active_members' => Member::active()->count(),
            'new_this_month' => Member::whereMonth('created_at', now()->month)->count(),
            'by_gender' => Member::selectRaw('gender, COUNT(*) as count')
                                ->groupBy('gender')
                                ->pluck('count', 'gender'),
            'by_age_group' => [
                'children' => Member::byAgeRange(0, 12)->count(),
                'youth' => Member::byAgeRange(13, 25)->count(),
                'adults' => Member::byAgeRange(26, 59)->count(),
                'seniors' => Member::byAgeRange(60, 120)->count(),
            ],
            'by_membership_status' => Member::selectRaw('membership_status, COUNT(*) as count')
                                          ->groupBy('membership_status')
                                          ->pluck('count', 'membership_status'),
        ];
        
        return response()->json($stats);
    }

    /**
     * Show the bulk import form
     */
    public function showBulkImport()
    {
        return view('members.bulk-import');
    }

    /**
     * Process bulk import from Excel file
     */
    public function processBulkImport(Request $request)
    {
        $request->validate([
            'excel_file' => 'required|file|mimes:xlsx,xls,csv|max:10240', // 10MB max
        ]);

        try {
            // Increase execution time for large imports
            set_time_limit(300); // 5 minutes
            ini_set('memory_limit', '512M'); // Increase memory limit
            
            $file = $request->file('excel_file');
            $import = new \App\Imports\MembersImport();
            
            Excel::import($import, $file);
            
            $results = $import->getResults();
            
            return redirect()->route('members.bulk-import')
                ->with('success', "Import completed! {$results['success']} members imported successfully. {$results['errors']} errors occurred.")
                ->with('import_results', $results);
                
        } catch (\Exception $e) {
            \Log::error('Bulk import failed: ' . $e->getMessage());
            return redirect()->route('members.bulk-import')
                ->with('error', 'Import failed: ' . $e->getMessage());
        }
    }

    /**
     * Download Excel template for bulk import
     */
    public function downloadTemplate()
    {
        return Excel::download(new \App\Exports\MemberTemplateExport(), 'member_import_template.xlsx');
    }

    /**
     * Reset member password (Admin only)
     */
    public function resetPassword(Request $request, Member $member)
    {
        $request->validate([
            'new_password' => 'required|string|min:8|confirmed',
        ]);

        $member->update([
            'password' => Hash::make($request->new_password),
        ]);

        return back()->with('success', 'Password reset successfully for ' . $member->full_name);
    }

    /**
     * Generate random password for member (Admin only)
     */
    public function generatePassword(Member $member)
    {
        $newPassword = Str::random(12);
        
        $member->update([
            'password' => Hash::make($newPassword),
        ]);

        return back()->with('success', 'New password generated for ' . $member->full_name)
                    ->with('generated_password', $newPassword);
    }

    /**
     * Disable member login access
     */
    public function disableLogin(Member $member)
    {
        $member->update([
            'password' => null,
        ]);

        return back()->with('success', 'Login access disabled for ' . $member->full_name);
    }

    /**
     * Enable member login access with temporary password
     */
    public function enableLogin(Member $member)
    {
        $tempPassword = 'temp' . rand(1000, 9999);
        
        $member->update([
            'password' => Hash::make($tempPassword),
        ]);

        return back()->with('success', 'Login access enabled for ' . $member->full_name)
                    ->with('temp_password', $tempPassword);
    }

    /**
     * Send registration email to member
     */
    private function sendMemberRegistrationEmail(Member $member)
    {
        try {
            if (!$member->email) {
                return;
            }

            // Use different email based on membership status
            // Active members (admin-created) get welcome email
            // Pending members (self-registered) get registration confirmation
            if ($member->membership_status === 'active') {
                Mail::to($member->email)->send(new MemberWelcomeEmail($member));
                \Log::info("Welcome email sent to {$member->email} for member {$member->id}");
            } else {
                Mail::to($member->email)->send(new MemberRegistrationEmail($member));
                \Log::info("Registration confirmation email sent to {$member->email} for member {$member->id}");
            }

            $member->update([
                'registration_email_sent_at' => now()
            ]);

        } catch (\Exception $e) {
            \Log::error('Member registration email failed: ' . $e->getMessage());
        }
    }

    /**
     * Send login credentials email to member
     */
    public function sendLoginCredentials(Member $member)
    {
        $request = request();
        $tempPassword = 'PCG' . rand(10000, 99999);

        try {
            // Set the password for the member
            $member->update([
                'password' => Hash::make($tempPassword)
            ]);

            if (!$member->email || !$member->canReceiveCommunication('email')) {
                return back()->with('error', 'Member does not have a valid email address.');
            }

            Mail::to($member->email)->send(new MemberLoginCredentialsEmail($member, $tempPassword));

            $member->update([
                'credentials_email_sent_at' => now()
            ]);

            return back()->with('success', 'Login credentials sent to ' . $member->email);

        } catch (\Exception $e) {
            \Log::error('Login credentials email failed: ' . $e->getMessage());
            return back()->with('error', 'Failed to send login credentials.');
        }
    }
}
