<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\InventoryCategory;
use App\Models\Setting;

class InventoryCategoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Get organization settings for dynamic content
        $organizationSettings = [
            'name' => Setting::getValue('organization_name', 'general', 'Presbyterian Church of Ghana'),
            'slogan' => Setting::getValue('organization_slogan', 'general', 'Church Management System'),
            'logo' => Setting::getValue('organization_logo', 'general')
        ];

        $query = InventoryCategory::withCount(['items', 'activeItems']);

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Filter by active status
        if ($request->filled('status')) {
            $query->where('is_active', $request->status === 'active');
        }

        $categories = $query->orderBy('name')->paginate(20);

        return view('inventory.categories.index', compact('categories', 'organizationSettings'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('inventory.categories.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:inventory_categories,name',
            'description' => 'nullable|string',
            'color' => 'required|string|size:7|regex:/^#[0-9A-Fa-f]{6}$/',
            'icon' => 'required|string|max:50'
        ]);

        InventoryCategory::create($validated);

        return redirect()->route('inventory.categories.index')
                        ->with('success', 'Category created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(InventoryCategory $category)
    {
        $category->load(['items' => function($query) {
            $query->with('recentTransactions')->orderBy('name');
        }]);

        return view('inventory.categories.show', compact('category'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(InventoryCategory $category)
    {
        return view('inventory.categories.edit', compact('category'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, InventoryCategory $category)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:inventory_categories,name,' . $category->id,
            'description' => 'nullable|string',
            'color' => 'required|string|size:7|regex:/^#[0-9A-Fa-f]{6}$/',
            'icon' => 'required|string|max:50',
            'is_active' => 'boolean'
        ]);

        $category->update($validated);

        return redirect()->route('inventory.categories.index')
                        ->with('success', 'Category updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(InventoryCategory $category)
    {
        // Check if category has items
        if ($category->items()->count() > 0) {
            return redirect()->back()
                           ->with('error', 'Cannot delete category that has items. Please move or delete all items first.');
        }

        $category->delete();

        return redirect()->route('inventory.categories.index')
                        ->with('success', 'Category deleted successfully!');
    }
}
