<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Document;
use App\Models\DocumentCategory;
use App\Models\Setting;
use App\Services\MNotifyService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }
    /**
     * Display the dashboard with key metrics and analytics.
     *
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user = auth()->user();
        
        // Check user role and redirect accordingly
        if ($user->role === User::ROLE_STAFF) {
            return redirect()->route('users.portal');
        }
        
        // Get SMS credits from MNotify
        $mnotifyService = new MNotifyService();
        $smsBalance = $mnotifyService->getBalance();
        $smsCredits = 0;
        
        if ($smsBalance['success']) {
            $smsCredits = $smsBalance['balance'];
        } else {
            // Log the error for debugging
            \Log::warning('Failed to fetch MNotify balance for dashboard', [
                'error' => $smsBalance['error'] ?? 'Unknown error',
                'response' => $smsBalance['response'] ?? null
            ]);
        }

        // Get church management statistics
        $churchStats = [
            'total_members' => \App\Models\Member::count(),
            'active_members' => \App\Models\Member::active()->count(),
            'total_families' => \App\Models\Family::count(),
            'total_ministries' => \App\Models\Ministry::active()->count(),
            'total_events' => \App\Models\Event::count(), // Changed from upcoming_events to total_events
            'sms_credits' => $smsCredits, // Added SMS credits from MNotify
            'total_donations_this_year' => \App\Models\Donation::confirmed()
                ->whereYear('donation_date', now()->year)
                ->sum('amount'),
            // Birthday statistics
            'total_birthdays' => \App\Models\Member::whereNotNull('date_of_birth')->count(),
            'birthdays_this_month' => \App\Models\Member::whereNotNull('date_of_birth')
                ->whereRaw('MONTH(date_of_birth) = ?', [now()->month])
                ->count(),
            'birthdays_this_week' => \App\Models\Member::whereNotNull('date_of_birth')
                ->whereRaw('WEEK(date_of_birth) = ?', [now()->week])
                ->count(),
            // Payment statistics  
            'total_payments' => \App\Models\Donation::confirmed()->count(),
            'payments_this_month' => \App\Models\Donation::confirmed()
                ->whereYear('donation_date', now()->year)
                ->whereMonth('donation_date', now()->month)
                ->count(),
        ];
        
        // Get legacy document management statistics (still available)
        $documentStats = [
            'total_users' => User::count(),
            'total_documents' => Document::count(),
            'total_categories' => DocumentCategory::count(),
            'total_prints' => \App\Models\UserDocumentPrint::sum('print_count'),
        ];

        // Legacy variables for backward compatibility
        $totalUsers = $documentStats['total_users'];
        $totalDocuments = $documentStats['total_documents'];
        $totalCategories = $documentStats['total_categories'];
        $totalPrints = $documentStats['total_prints'];
        
        // Additional legacy variables
        $userGrowth = 0; // Placeholder for growth calculation
        $activeDocuments = Document::count(); // All documents are considered active
        $documentGrowth = 0; // Placeholder for growth calculation
        $totalStorageBytes = Document::sum('file_size') ?? 0;
        $averageFileSize = $totalDocuments > 0 ? round(($totalStorageBytes / 1024 / 1024) / $totalDocuments, 1) : 0;
        
        // Get recent documents for legacy dashboard
        $recentDocuments = Document::with('category')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
            
        // Get top printed documents for legacy dashboard
        $topPrintedDocuments = Document::with('category')
            ->where('print_count', '>', 0)
            ->orderBy('print_count', 'desc')
            ->limit(6)
            ->get();
        
        // Get recent church activities
        $recentMembers = \App\Models\Member::with('family')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
            
        $upcomingEvents = \App\Models\Event::with(['ministry', 'organizer'])
            ->upcoming()
            ->published()
            ->orderBy('start_datetime')
            ->limit(5)
            ->get();
            
        $recentDonations = \App\Models\Donation::with('member')
            ->confirmed()
            ->orderBy('donation_date', 'desc')
            ->limit(5)
            ->get();
        
        // Get recent activities (legacy)
        $recentActivities = \App\Models\ActivityLog::with('user')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
        
        // Get monthly donation statistics
        $monthlyDonations = \App\Models\Donation::selectRaw('MONTH(donation_date) as month, SUM(amount) as total')
            ->confirmed()
            ->whereYear('donation_date', now()->year)
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->pluck('total', 'month')
            ->toArray();
        
        // Fill missing months with 0
        for ($i = 1; $i <= 12; $i++) {
            if (!isset($monthlyDonations[$i])) {
                $monthlyDonations[$i] = 0;
            }
        }
        ksort($monthlyDonations);
        
        // Get membership growth statistics (current year)
        $membershipGrowth = \App\Models\Member::selectRaw('MONTH(created_at) as month, COUNT(*) as total')
            ->whereYear('created_at', now()->year)
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->pluck('total', 'month')
            ->toArray();
            
        // Fill missing months with 0
        for ($i = 1; $i <= 12; $i++) {
            if (!isset($membershipGrowth[$i])) {
                $membershipGrowth[$i] = 0;
            }
        }
        ksort($membershipGrowth);
        
        // Get membership growth by year (last 5 years)
        $yearlyMembershipGrowth = \App\Models\Member::selectRaw('YEAR(created_at) as year, COUNT(*) as total')
            ->where('created_at', '>=', now()->subYears(5))
            ->groupBy('year')
            ->orderBy('year')
            ->get()
            ->pluck('total', 'year')
            ->toArray();
            
        // Get cumulative membership count by month (current year)
        $cumulativeMembership = [];
        $totalMembers = \App\Models\Member::where('created_at', '<', now()->startOfYear())->count();
        
        for ($i = 1; $i <= 12; $i++) {
            $monthlyNew = $membershipGrowth[$i] ?? 0;
            $totalMembers += $monthlyNew;
            $cumulativeMembership[$i] = $totalMembers;
        }
        
        // Get member demographics data
        $genderStats = \App\Models\Member::selectRaw('gender, COUNT(*) as count')
            ->where('membership_status', 'active')
            ->groupBy('gender')
            ->pluck('count', 'gender')
            ->toArray();
            
        $maritalStats = \App\Models\Member::selectRaw('marital_status, COUNT(*) as count')
            ->where('membership_status', 'active')
            ->whereNotNull('marital_status')
            ->groupBy('marital_status')
            ->pluck('count', 'marital_status')
            ->toArray();
        
        // Get PCG Year Group demographics
        $yearGroupStats = \App\Models\YearGroup::active()
            ->ordered()
            ->withCount(['members', 'members as active_members_count' => function ($query) {
                $query->where('membership_status', 'active');
            }])
            ->get();
        
        // Get additional member statistics
        $memberStats = [
            'new_members_this_month' => \App\Models\Member::whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->count(),
            'new_members_this_year' => \App\Models\Member::whereYear('created_at', now()->year)->count(),
            'baptized_members' => \App\Models\Member::where('is_baptized', true)->count(),
            'confirmed_members' => \App\Models\Member::where('is_confirmed', true)->count(),
            'married_members' => \App\Models\Member::where('marital_status', 'married')->count(),
            'members_with_email' => \App\Models\Member::whereNotNull('email')->count(),
            'members_with_phone' => \App\Models\Member::whereNotNull('phone')->count(),
            'unassigned_year_group' => \App\Models\Member::whereNull('year_group_id')->count(),
        ];
        
        // Get ministry engagement statistics
        $ministryStats = [
            'total_ministry_members' => \DB::table('member_ministry')
                ->where('member_ministry.is_active', true)
                ->distinct('member_id')
                ->count('member_id'),
            'members_without_ministry' => \App\Models\Member::whereDoesntHave('ministries', function ($query) {
                $query->where('member_ministry.is_active', true);
            })->count(),
            'average_members_per_ministry' => \App\Models\Ministry::active()
                ->withCount(['members' => function ($query) {
                    $query->where('member_ministry.is_active', true);
                }])
                ->get()
                ->avg('members_count'),
        ];
        
        // Legacy age demographics for backward compatibility
        $ageDemographics = [
            'children' => \App\Models\Member::byAgeRange(0, 12)->count(),
            'youth' => \App\Models\Member::byAgeRange(13, 25)->count(),
            'adults' => \App\Models\Member::byAgeRange(26, 59)->count(),
            'seniors' => \App\Models\Member::byAgeRange(60, 120)->count(),
        ];
        
        // Get organization settings for dynamic content
        $organizationSettings = [
            'name' => Setting::getValue('organization_name', 'general', 'Presbyterian Church of Ghana'),
            'slogan' => Setting::getValue('organization_slogan', 'general', 'Church Management System'),
            'logo' => Setting::getValue('organization_logo', 'general'),
        ];
        
        return view('dashboard', compact(
            'churchStats',
            'documentStats', 
            'recentActivities', 
            'recentMembers',
            'upcomingEvents',
            'recentDonations',
            'monthlyDonations',
            'membershipGrowth',
            'yearlyMembershipGrowth',
            'cumulativeMembership',
            'genderStats',
            'maritalStats',
            'ageDemographics',
            'yearGroupStats',
            'memberStats',
            'ministryStats',
            'totalUsers',
            'totalDocuments',
            'totalCategories',
            'totalPrints',
            'userGrowth',
            'activeDocuments',
            'documentGrowth',
            'totalStorageBytes',
            'averageFileSize',
            'recentDocuments',
            'topPrintedDocuments',
            'organizationSettings'
        ));
    }
}
