<?php

namespace App\Http\Controllers;

use App\Models\BirthdayMessage;
use App\Models\Member;
use App\Services\MNotifyService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class BirthdayMessageController extends Controller
{
    protected $mnotifyService;

    public function __construct(MNotifyService $mnotifyService)
    {
        $this->mnotifyService = $mnotifyService;
    }

    /**
     * Display a listing of birthday messages
     */
    public function index(Request $request)
    {
        $query = BirthdayMessage::with('member')->orderBy('created_at', 'desc');

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->whereDate('birthday_date', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('birthday_date', '<=', $request->date_to);
        }

        $messages = $query->paginate(20);

        // Get statistics
        $stats = [
            'total_messages' => BirthdayMessage::count(),
            'sent_messages' => BirthdayMessage::sent()->count(),
            'pending_messages' => BirthdayMessage::pending()->count(),
            'failed_messages' => BirthdayMessage::failed()->count(),
            'today_birthdays' => $this->getTodayBirthdays()->count(),
            'this_week_birthdays' => $this->getThisWeekBirthdays()->count(),
        ];

        return view('birthday-messages.index', compact('messages', 'stats'));
    }

    /**
     * Show today's birthdays
     */
    public function todayBirthdays()
    {
        $todayBirthdays = $this->getTodayBirthdays();
        $sentMessages = BirthdayMessage::today()->sent()->pluck('member_id')->toArray();

        return view('birthday-messages.today', compact('todayBirthdays', 'sentMessages'));
    }

    /**
     * Generate birthday messages for today
     */
    public function generateTodayMessages(Request $request)
    {
        $todayBirthdays = $this->getTodayBirthdays();
        $messageType = $request->input('message_type', 'sms');
        $customMessage = $request->input('custom_message');
        
        $generated = 0;
        $skipped = 0;

        foreach ($todayBirthdays as $member) {
            // Check if message already exists for today
            $existingMessage = BirthdayMessage::where('member_id', $member->id)
                ->whereDate('birthday_date', Carbon::today())
                ->first();

            if ($existingMessage) {
                $skipped++;
                continue;
            }

            // Generate personalized message
            $message = $customMessage ?: $this->generateBirthdayMessage($member);

            // Create birthday message record
            BirthdayMessage::create([
                'member_id' => $member->id,
                'message_type' => $messageType,
                'message_content' => $message,
                'phone_number' => $member->phone,
                'email_address' => $member->email,
                'birthday_date' => Carbon::today(),
                'status' => 'pending'
            ]);

            $generated++;
        }

        return redirect()->route('birthday-messages.today')
            ->with('success', "Generated {$generated} birthday messages. Skipped {$skipped} existing messages.");
    }

    /**
     * Send birthday messages
     */
    public function sendMessages(Request $request)
    {
        $messageIds = $request->input('message_ids', []);
        $messages = BirthdayMessage::whereIn('id', $messageIds)->pending()->get();

        $sent = 0;
        $failed = 0;

        foreach ($messages as $message) {
            try {
                if ($message->message_type === 'sms' || $message->message_type === 'both') {
                    if ($message->phone_number) {
                        $result = $this->mnotifyService->sendSMS($message->phone_number, $message->message_content);
                        
                        if ($result['success']) {
                            $message->markAsSent($result);
                            $sent++;
                        } else {
                            $message->markAsFailed($result['error'] ?? 'SMS sending failed');
                            $failed++;
                        }
                    } else {
                        $message->markAsFailed('No phone number available');
                        $failed++;
                    }
                }

                if ($message->message_type === 'email' || $message->message_type === 'both') {
                    if ($message->email_address) {
                        // Send email (implement email sending logic)
                        // Mail::to($message->email_address)->send(new BirthdayEmail($message));
                        // For now, mark as sent
                        $message->markAsSent(['type' => 'email']);
                        $sent++;
                    } else {
                        $message->markAsFailed('No email address available');
                        $failed++;
                    }
                }
            } catch (\Exception $e) {
                $message->markAsFailed($e->getMessage());
                $failed++;
            }
        }

        return redirect()->back()
            ->with('success', "Sent {$sent} messages successfully. {$failed} messages failed.");
    }

    /**
     * Get today's birthdays
     */
    private function getTodayBirthdays()
    {
        return Member::whereNotNull('date_of_birth')
            ->whereRaw('DATE_FORMAT(date_of_birth, "%m-%d") = ?', [Carbon::today()->format('m-d')])
            ->where('membership_status', 'active')
            ->get();
    }

    /**
     * Get this week's birthdays
     */
    private function getThisWeekBirthdays()
    {
        $startOfWeek = Carbon::now()->startOfWeek()->format('m-d');
        $endOfWeek = Carbon::now()->endOfWeek()->format('m-d');

        return Member::whereNotNull('date_of_birth')
            ->where('membership_status', 'active')
            ->whereRaw('DATE_FORMAT(date_of_birth, "%m-%d") BETWEEN ? AND ?', [$startOfWeek, $endOfWeek])
            ->get();
    }

    /**
     * Generate personalized birthday message
     */
    private function generateBirthdayMessage(Member $member): string
    {
        $firstName = explode(' ', $member->full_name)[0];
        $age = $member->age;
        $yearGroup = $member->yearGroup ? $member->yearGroup->name : '';

        $messages = [
            "🎉 Happy Birthday {$firstName}! May God bless you with joy, peace, and prosperity as you celebrate another year of life. Enjoy your special day! - Presbyterian Church of Ghana",
            "🎂 Wishing you a very Happy Birthday {$firstName}! May this new year of your life be filled with God's abundant blessings and grace. Have a wonderful celebration!",
            "🌟 Happy Birthday {$firstName}! On your special day, we pray that God continues to guide your steps and shower you with His love. Celebrate and enjoy! - Your Church Family",
            "🎈 It's your birthday {$firstName}! May God grant you health, happiness, and success in the year ahead. Have a blessed and joyful celebration!",
        ];

        $message = $messages[array_rand($messages)];

        // Add age-specific message if available
        if ($age) {
            $message = str_replace('!', " on your {$age}th birthday!", $message);
        }

        return $message;
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $members = Member::where('membership_status', 'active')
            ->whereNotNull('date_of_birth')
            ->orderBy('first_name')
            ->orderBy('last_name')
            ->get();

        return view('birthday-messages.create', compact('members'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'member_ids' => 'required|array',
            'member_ids.*' => 'exists:members,id',
            'message_type' => 'required|in:sms,email,both',
            'message_content' => 'required|string|max:1000',
            'send_date' => 'required|date|after_or_equal:today'
        ]);

        $created = 0;
        foreach ($request->member_ids as $memberId) {
            $member = Member::find($memberId);
            
            BirthdayMessage::create([
                'member_id' => $memberId,
                'message_type' => $request->message_type,
                'message_content' => $request->message_content,
                'phone_number' => $member->phone,
                'email_address' => $member->email,
                'birthday_date' => $request->send_date,
                'status' => 'pending'
            ]);
            
            $created++;
        }

        return redirect()->route('birthday-messages.index')
            ->with('success', "Created {$created} birthday messages successfully.");
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(BirthdayMessage $birthdayMessage)
    {
        $birthdayMessage->delete();

        return redirect()->route('birthday-messages.index')
            ->with('success', 'Birthday message deleted successfully.');
    }
}
