<?php

namespace App\Console\Commands;

use App\Models\Member;
use App\Models\BirthdayMessage;
use App\Services\MNotifyService;
use Carbon\Carbon;
use Illuminate\Console\Command;

class SendBirthdayMessages extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'birthday:send {--dry-run : Show what would be sent without actually sending}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Automatically send birthday messages to members celebrating today';

    protected $mnotifyService;

    public function __construct(MNotifyService $mnotifyService)
    {
        parent::__construct();
        $this->mnotifyService = $mnotifyService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🎂 Starting Birthday Message Sender...');
        
        // Get today's birthday celebrants
        $todayBirthdays = Member::whereNotNull('date_of_birth')
            ->whereRaw('DATE_FORMAT(date_of_birth, "%m-%d") = ?', [Carbon::today()->format('m-d')])
            ->where('membership_status', 'active')
            ->get();

        if ($todayBirthdays->isEmpty()) {
            $this->info('✅ No birthdays today. Nothing to send.');
            return 0;
        }

        $this->info("🎉 Found {$todayBirthdays->count()} birthday(s) today:");
        
        $generated = 0;
        $sent = 0;
        $failed = 0;
        $skipped = 0;

        foreach ($todayBirthdays as $member) {
            $this->line("   - {$member->full_name} (Age: {$member->age})");
            
            // Check if message already exists for today
            $existingMessage = BirthdayMessage::where('member_id', $member->id)
                ->whereDate('birthday_date', Carbon::today())
                ->first();

            if ($existingMessage) {
                if ($existingMessage->status === 'sent') {
                    $this->warn("     ⏭️  Message already sent to {$member->full_name}");
                    $skipped++;
                    continue;
                } elseif ($existingMessage->status === 'failed') {
                    $this->warn("     🔄 Retrying failed message for {$member->full_name}");
                    $message = $existingMessage;
                } else {
                    $this->info("     📤 Sending pending message to {$member->full_name}");
                    $message = $existingMessage;
                }
            } else {
                // Generate new birthday message
                $messageContent = $this->generateBirthdayMessage($member);
                
                if ($this->option('dry-run')) {
                    $this->info("     📝 Would create message: {$messageContent}");
                    $generated++;
                    continue;
                }

                $message = BirthdayMessage::create([
                    'member_id' => $member->id,
                    'message_type' => 'sms',
                    'message_content' => $messageContent,
                    'phone_number' => $member->phone,
                    'email_address' => $member->email,
                    'birthday_date' => Carbon::today(),
                    'status' => 'pending'
                ]);
                
                $generated++;
                $this->info("     ✅ Generated message for {$member->full_name}");
            }

            // Send the message if not in dry-run mode
            if (!$this->option('dry-run')) {
                if ($member->phone) {
                    try {
                        $result = $this->mnotifyService->sendSMS($member->phone, $message->message_content);
                        
                        if ($result['success']) {
                            $message->markAsSent($result);
                            $this->info("     📱 SMS sent successfully to {$member->full_name}");
                            $sent++;
                        } else {
                            $message->markAsFailed($result['error'] ?? 'SMS sending failed');
                            $this->error("     ❌ Failed to send SMS to {$member->full_name}: " . ($result['error'] ?? 'Unknown error'));
                            $failed++;
                        }
                    } catch (\Exception $e) {
                        $message->markAsFailed($e->getMessage());
                        $this->error("     ❌ Exception sending to {$member->full_name}: " . $e->getMessage());
                        $failed++;
                    }
                } else {
                    $message->markAsFailed('No phone number available');
                    $this->warn("     ⚠️  No phone number for {$member->full_name}");
                    $failed++;
                }
            }
        }

        // Summary
        $this->newLine();
        $this->info('📊 Summary:');
        
        if ($this->option('dry-run')) {
            $this->line("   🔍 Dry run completed");
            $this->line("   📝 Would generate: {$generated} messages");
            $this->line("   ⏭️  Already processed: {$skipped} members");
        } else {
            $this->line("   📝 Generated: {$generated} new messages");
            $this->line("   📱 Sent: {$sent} messages");
            $this->line("   ❌ Failed: {$failed} messages");
            $this->line("   ⏭️  Skipped: {$skipped} messages");
        }

        $this->newLine();
        $this->info('🎂 Birthday message sending completed!');
        
        return 0;
    }

    /**
     * Generate personalized birthday message
     */
    private function generateBirthdayMessage(Member $member): string
    {
        $firstName = explode(' ', $member->full_name)[0];
        $age = $member->age;

        $messages = [
            "🎉 Happy Birthday {$firstName}! May God bless you with joy, peace, and prosperity as you celebrate another year of life. Enjoy your special day! - Presbyterian Church of Ghana",
            "🎂 Wishing you a very Happy Birthday {$firstName}! May this new year of your life be filled with God's abundant blessings and grace. Have a wonderful celebration!",
            "🌟 Happy Birthday {$firstName}! On your special day, we pray that God continues to guide your steps and shower you with His love. Celebrate and enjoy! - Your Church Family",
            "🎈 It's your birthday {$firstName}! May God grant you health, happiness, and success in the year ahead. Have a blessed and joyful celebration!",
        ];

        $message = $messages[array_rand($messages)];

        // Add age-specific message if available
        if ($age) {
            $message = str_replace('!', " on your {$age}th birthday!", $message);
        }

        return $message;
    }
}
