<?php

namespace App\Console\Commands;

use App\Models\Member;
use App\Mail\BirthdayEmail;
use App\Services\MNotifyService;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class SendBirthdayEmailsCommand extends Command
{
    protected $signature = 'birthday:send-emails {--dry-run : Show what would be sent without actually sending}';
    protected $description = 'Send birthday emails and SMS to members celebrating today';

    protected $mnotifyService;

    public function __construct(MNotifyService $mnotifyService)
    {
        parent::__construct();
        $this->mnotifyService = $mnotifyService;
    }

    public function handle()
    {
        $isDryRun = $this->option('dry-run');
        $this->info('🎂 Starting Birthday Email & SMS Sender...');
        
        if ($isDryRun) {
            $this->warn('DRY RUN MODE - No emails or SMS will be sent');
        }

        // Get today's birthday celebrants
        $todayBirthdays = Member::whereNotNull('date_of_birth')
            ->whereRaw('DATE_FORMAT(date_of_birth, "%m-%d") = ?', [Carbon::today()->format('m-d')])
            ->where('is_active', true)
            ->get();

        if ($todayBirthdays->isEmpty()) {
            $this->info('✅ No birthdays today.');
            return 0;
        }

        $this->info("🎉 Found {$todayBirthdays->count()} birthday(s) today:");
        
        $emailsSent = 0;
        $smsSent = 0;
        $emailsFailed = 0;
        $smsFailed = 0;
        $skipped = 0;

        foreach ($todayBirthdays as $member) {
            $this->line("   Processing: {$member->full_name} (Age: {$member->age})");
            
            // Check if already sent today
            if ($member->last_birthday_email_sent_at && 
                $member->last_birthday_email_sent_at->isToday()) {
                $this->warn("     ⏭️  Birthday greeting already sent today");
                $skipped++;
                continue;
            }

            // Send Email
            if ($member->email && $member->canReceiveCommunication('email')) {
                try {
                    if (!$isDryRun) {
                        Mail::to($member->email)->send(new BirthdayEmail($member));
                        $this->info("     ✅ Email sent to {$member->email}");
                        $emailsSent++;
                    } else {
                        $this->info("     📧 Would send email to {$member->email}");
                        $emailsSent++;
                    }
                } catch (\Exception $e) {
                    $this->error("     ❌ Email failed: " . $e->getMessage());
                    Log::error("Birthday email failed for member {$member->id}: " . $e->getMessage());
                    $emailsFailed++;
                }
            }

            // Send SMS
            if ($member->phone && $member->canReceiveCommunication('sms')) {
                try {
                    $message = $this->generateBirthdaySmsMessage($member);
                    
                    if (!$isDryRun) {
                        $result = $this->mnotifyService->sendSms($member->phone, $message);
                        
                        if ($result['success']) {
                            $this->info("     📱 SMS sent to {$member->phone}");
                            $smsSent++;
                        } else {
                            $this->error("     ❌ SMS failed: " . ($result['message'] ?? 'Unknown error'));
                            $smsFailed++;
                        }
                    } else {
                        $this->info("     📱 Would send SMS to {$member->phone}");
                        $smsSent++;
                    }
                } catch (\Exception $e) {
                    $this->error("     ❌ SMS exception: " . $e->getMessage());
                    Log::error("Birthday SMS failed for member {$member->id}: " . $e->getMessage());
                    $smsFailed++;
                }
            }

            // Update last sent timestamp
            if (!$isDryRun && ($emailsSent > 0 || $smsSent > 0)) {
                $member->update(['last_birthday_email_sent_at' => now()]);
            }

            // Small delay to avoid overwhelming servers
            if (!$isDryRun) {
                sleep(1);
            }
        }

        // Summary
        $this->newLine();
        $this->info('📊 Birthday Greetings Summary:');
        $this->line("   📧 Emails: {$emailsSent} sent, {$emailsFailed} failed");
        $this->line("   📱 SMS: {$smsSent} sent, {$smsFailed} failed");
        $this->line("   ⏭️  Skipped: {$skipped}");
        
        $this->newLine();
        $this->info('🎂 Birthday greetings completed!');
        
        return 0;
    }

    private function generateBirthdaySmsMessage(Member $member): string
    {
        $firstName = explode(' ', $member->full_name)[0];
        $age = $member->age;
        $organizationName = \DB::table('settings')
            ->where('key', 'organization_name')
            ->value('value') ?? 'Presbyterian Church of Ghana';

        $messages = [
            "🎉 Happy Birthday {$firstName}! May God bless you with joy, peace, and prosperity as you celebrate another year. Enjoy your special day! - {$organizationName}",
            "🎂 Wishing you a blessed birthday {$firstName}! May this new year be filled with God's abundant grace and blessings. Have a wonderful celebration!",
            "🌟 Happy Birthday {$firstName}! We celebrate you today and pray for God's continued guidance in your life. Have a blessed day! - {$organizationName}",
        ];

        return $messages[array_rand($messages)];
    }
}
