# Bulk Import Auto-Assignment Guide

## Overview
This guide explains how member data is automatically assigned during bulk import and how to fix existing unassigned members.

## What Gets Auto-Assigned During Import?

### 1. **Year Group Assignment** ✅
- **Field:** `year_group_id`
- **How it works:** Automatically calculated based on member's age from `date_of_birth`
- **Requirements:** 
  - Member must have a valid `date_of_birth`
  - A matching year group must exist for that age range
  - Year groups must be active

### 2. **Family Assignment** ✅
- **Field:** `family_id`
- **How it works:** Members with the same `family_name` in Excel are grouped together
- **Auto-creates:** If family doesn't exist, it's created automatically

### 3. **Member ID** ✅
- **Field:** `member_id`
- **Format:** `PCG0001`, `PCG0002`, etc.
- **Auto-generated:** Sequential numbering starting from last member ID

### 4. **Day Born** ✅
- **Field:** `day_born`
- **Values:** Sunday, Monday, Tuesday, etc.
- **Calculated from:** `date_of_birth`

### 5. **Default Values** ✅
The following fields are auto-filled if not provided in Excel:
- `membership_type` → `communicant`
- `membership_status` → `active`
- `membership_date` → Current date
- `marital_status` → `single`
- `password` → `password123` (default login password)
- `is_active` → `true`
- `receive_newsletter` → `true`
- `receive_sms` → `true`

## Excel Template Columns

### Required Columns:
- `first_name`
- `last_name`
- `phone`
- `gender` (male/female)

### Optional Columns:
- `middle_name`
- `email`
- `date_of_birth` (YYYY-MM-DD format)
- `membership_type` (communicant/baptized/confirmed/long_distance/adherent/catechumen)
- `membership_status` (active/inactive/transferred/deceased)
- `marital_status` (single/married/divorced/widowed)
- `address`
- `occupation`
- `emergency_contact_name`
- `emergency_contact_phone`
- `family_name` (for family grouping)

## Fixing Existing Unassigned Members

### Command: Assign Year Groups
For members imported before auto-assignment was implemented:

```bash
# Assign year groups to members without assignment
php artisan members:assign-year-groups

# Force reassignment for ALL members (updates everyone)
php artisan members:assign-year-groups --force
```

### What This Does:
1. ✅ Finds all members with `date_of_birth` but no `year_group_id`
2. ✅ Calculates their current age
3. ✅ Finds matching year group based on age range
4. ✅ Assigns the appropriate year group
5. ✅ Shows summary report

### Example Output:
```
🔄 Starting year group assignment...
Processing members without year group assignment...
Found 992 member(s) to process.
 992/992 [▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓▓] 100%

✅ Year group assignment completed!
+-------------------+-------+
| Status            | Count |
+-------------------+-------+
| Assigned          | 992   |
| No matching group | 0     |
| Skipped (errors)  | 0     |
+-------------------+-------+
```

## Troubleshooting

### Issue: Members Not Getting Year Groups
**Possible Causes:**
1. **No Year Groups Defined**
   - Solution: Create year groups via Settings → Year Groups
   - Example: Children (0-12), Youth (13-18), Adults (19-59), Seniors (60+)

2. **Age Outside Defined Ranges**
   - Solution: Create year groups that cover all age ranges
   - Check: Are there gaps in your age ranges?

3. **Missing Date of Birth**
   - Solution: Members need valid `date_of_birth` for auto-assignment
   - Update member records to include DOB

### Issue: Duplicate Phone Numbers/Emails
**Error:** "Member with this phone number or email already exists"
**Solution:** 
- Check Excel for duplicates
- Verify existing database for conflicts
- Use unique phone numbers and emails

### Issue: Invalid Data Format
**Common Errors:**
- Date format must be: `YYYY-MM-DD` (e.g., `1990-01-15`)
- Gender must be: `male` or `female`
- Phone must be unique
- Email must be valid format

## Best Practices

### 1. Prepare Year Groups First
Before bulk import, ensure year groups are configured:
- Navigate to: Settings → Year Groups
- Create age-appropriate groups
- Ensure no gaps in age ranges

### 2. Clean Your Excel Data
- Remove duplicate phone numbers
- Use consistent date format
- Validate email addresses
- Check gender values

### 3. Test with Small Batch
- Import 5-10 members first
- Verify auto-assignments work
- Check year groups assigned correctly
- Then proceed with full import

### 4. Regular Maintenance
Run yearly to update age-based assignments:
```bash
php artisan members:assign-year-groups --force
```

## Access Points

- **Bulk Import Page:** http://127.0.0.1:8000/members/bulk-import
- **Download Template:** Available on bulk import page
- **Members List:** http://127.0.0.1:8000/members
- **Year Groups:** http://127.0.0.1:8000/settings (Settings → Year Groups)

## Support

If members still show as "unassigned" after following this guide:
1. Verify year groups exist and are active
2. Run the assignment command
3. Check member has valid date of birth
4. Review error logs in `storage/logs/laravel.log`
